import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';

@Injectable()
export class AdvancedPayoutService {
  private logger = new Logger('AdvancedPayoutService');

  constructor(private prisma: PrismaService) {}

  // স্বয়ংক্রিয় পেমেন্ট প্রসেসিং (কনফিগারেশন অনুযায়ী)
  @Cron('0 10 * * *') // প্রতিদিন 10 AM
  async processScheduledPayouts() {
    const configs = await this.prisma.commissionConfig.findMany();

    for (const config of configs) {
      if (config.payoutFrequency === 'daily') {
        await this.processTenantPayouts(config.tenantId);
      }
    }
  }

  // সাপ্তাহিক পেমেন্ট (সোমবার)
  @Cron('0 10 * * 1')
  async processWeeklyPayouts() {
    const configs = await this.prisma.commissionConfig.findMany({
      where: { payoutFrequency: 'weekly' }
    });

    for (const config of configs) {
      await this.processTenantPayouts(config.tenantId);
    }
  }

  // মাসিক পেমেন্ট (প্রথম দিন)
  @Cron('0 10 1 * *')
  async processMonthlyPayouts() {
    const configs = await this.prisma.commissionConfig.findMany({
      where: { payoutFrequency: 'monthly' }
    });

    for (const config of configs) {
      await this.processTenantPayouts(config.tenantId);
    }
  }

  private async processTenantPayouts(tenantId: string) {
    const config = await this.prisma.commissionConfig.findUnique({
      where: { tenantId }
    });

    if (!config) return;

    const affiliates = await this.prisma.affiliateAccount.findMany({
      where: { tenantId, status: 'active' }
    });

    for (const affiliate of affiliates) {
      // মিনিমাম পেমেন্ট এমাউন্ট চেক করুন
      const balance = await this.getAvailableBalance(affiliate.id);

      if (balance >= Number(config.minimumPayoutAmount)) {
        if (affiliate.paymentMethod === 'auto') {
          // স্বয়ংক্রিয় পেমেন্ট
          await this.processAutoPayout(affiliate);
        }
        // ম্যানুয়াল পেমেন্ট এর জন্য ফ্ল্যাগ সেট করুন
      }
    }
  }

  // মিনিমাম এমাউন্ট চেক সহ পেমেন্ট রিকোয়েস্ট
  async requestPayoutWithValidation(affiliateId: string, amount: number) {
    const config = await this.prisma.commissionConfig.findFirst({
      where: {
        affiliate: {
          some: { id: affiliateId }
        }
      }
    });

    if (!config) throw new Error('Config not found');

    if (amount < Number(config.minimumPayoutAmount)) {
      throw new Error(
        `Minimum payout amount is $${config.minimumPayoutAmount}`
      );
    }

    const balance = await this.getAvailableBalance(affiliateId);

    if (balance < amount) {
      throw new Error('Insufficient balance');
    }

    // পেমেন্ট রিকোয়েস্ট তৈরি করুন
    const payout = await this.prisma.affiliatePayoutRequest.create({
      data: {
        affiliateId,
        amount,
        status: 'pending'
      }
    });

    return { success: true, payoutId: payout.id };
  }

  // ম্যানুয়াল পেমেন্ট অনুমোদন
  async approvePayoutManually(payoutId: string, adminNotes?: string) {
    const payout = await this.prisma.affiliatePayoutRequest.update({
      where: { id: payoutId },
      data: { status: 'processing' }
    });

    // প্রসেসিং জব কিউতে যোগ করুন
    // await this.payoutQueue.add('process-payout', { payoutId });

    return { success: true, data: payout };
  }

  // অ-অনুমোদিত পেমেন্ট প্রত্যাখ্যান
  async rejectPayout(payoutId: string, reason: string) {
    const payout = await this.prisma.affiliatePayoutRequest.update({
      where: { id: payoutId },
      data: { status: 'failed' }
    });

    // কমিশন ফিরিয়ে দিন
    await this.prisma.affiliateCommission.updateMany({
      where: {
        affiliateId: payout.affiliateId,
        status: 'approved'
      },
      data: { status: 'pending' }
    });

    return { success: true, data: payout };
  }

  private async getAvailableBalance(affiliateId: string): Promise<number> {
    const commissions = await this.prisma.affiliateCommission.findMany({
      where: { affiliateId, status: 'approved' }
    });

    return commissions.reduce((sum, c) => sum + Number(c.amount), 0);
  }

  private async processAutoPayout(affiliate: any) {
    // স্বয়ংক্রিয় পেমেন্ট লজিক
    this.logger.log(`Auto payout for affiliate: ${affiliate.id}`);
  }
}