import {
  Controller,
  Post,
  Get,
  Body,
  Param,
  Query,
  UseGuards,
  Req,
  Redirect
} from '@nestjs/common';
import { ApiTags, ApiBearerAuth } from '@nestjs/swagger';
import { AffiliateService } from './affiliate.service';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';

@ApiTags('Affiliates')
@Controller('affiliates')
export class AffiliateController {
  constructor(private affiliateService: AffiliateService) {}

  @Post('links')
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  async createLink(@Req() req: any, @Body() body: { productId?: string }) {
    return await this.affiliateService.generateAffiliateLink(
      req.user.tenantId,
      req.user.sub,
      body.productId
    );
  }

  @Get('links')
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  async listLinks(@Req() req: any, @Query('skip') skip = 0, @Query('take') take = 20) {
    return await this.affiliateService.listAffiliateLinks(req.user.sub, skip, take);
  }

  @Get('dashboard')
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  async getDashboard(@Req() req: any) {
    return await this.affiliateService.getAffiliateDashboard(req.user.sub);
  }

  @Get('track/:code')
  @Redirect()
  async trackClick(
    @Param('code') code: string,
    @Req() req: any
  ) {
    await this.affiliateService.trackAffiliateClick(
      code,
      req.ip,
      req.headers['user-agent']
    );

    // Redirect to home or product page
    return { url: `${process.env.APP_URL}` };
  }

  @Post('conversion')
  async trackConversion(@Body() body: { code: string; orderId: string; orderTotal: number }) {
    return await this.affiliateService.trackAffiliateConversion(
      body.code,
      body.orderId,
      body.orderTotal
    );
  }

  @Post('payout')
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  async requestPayout(
    @Req() req: any,
    @Body() body: { amount: number; paymentMethod: string }
  ) {
    return await this.affiliateService.requestPayout(req.user.sub, body.amount, body.paymentMethod);
  }

  @Get('report')
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  async getReport(
    @Req() req: any,
    @Query('startDate') startDate: string,
    @Query('endDate') endDate: string
  ) {
    return await this.affiliateService.getAffiliateReport(
      req.user.sub,
      new Date(startDate),
      new Date(endDate)
    );
  }
}