import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as crypto from 'crypto';
import * as QRCode from 'qrcode';

@Injectable()
export class AffiliateService {
  private logger = new Logger('AffiliateService');

  constructor(private prisma: PrismaService) {}

  // অ্যাফিলিয়েট অ্যাকাউন্ট তৈরি করুন (সব রোলের জন্য)
  async createAffiliateAccount(
    userId: string,
    tenantId: string,
    input: {
      commissionRate: number;
      paymentMethod?: string;
      paymentAccount?: string;
    },
    referrerId?: string // Parent affiliate (3-tier system)
  ) {
    try {
      // ইউজার চেক করুন
      const user = await this.prisma.user.findUnique({
        where: { id: userId }
      });

      if (!user) throw new BadRequestException('User not found');

      // Referrer চেক করুন এবং chain তৈরি করুন
      let referrerChain: string[] = [];
      if (referrerId) {
        const referrer = await this.prisma.affiliateAccount.findUnique({
          where: { id: referrerId }
        });

        if (!referrer) throw new BadRequestException('Referrer not found');

        // 3-tier chain তৈরি করুন
        referrerChain = [referrerId, ...referrer.referrerChain.slice(0, 2)];
      }

      // অ্যাফিলিয়েট অ্যাকাউন্ট তৈরি করুন
      const affiliate = await this.prisma.affiliateAccount.create({
        data: {
          userId,
          tenantId,
          role: user.role || 'user',
          referrerId,
          referrerChain,
          commissionRate: input.commissionRate,
          paymentMethod: input.paymentMethod,
          paymentAccount: input.paymentAccount ? this.encryptPaymentInfo(input.paymentAccount) : null
        }
      });

      // Stats রেকর্ড তৈরি করুন
      await this.prisma.affiliateStats.create({
        data: { affiliateId: affiliate.id }
      });

      this.logger.log(`Affiliate account created: ${affiliate.id} for user ${userId}`);

      return {
        success: true,
        data: {
          affiliateId: affiliate.id,
          role: affiliate.role,
          commissionRate: affiliate.commissionRate
        }
      };
    } catch (err) {
      this.logger.error('Affiliate creation failed:', err);
      throw err;
    }
  }

  // অ্যাফিলিয়েট অ্যাকাউন্ট অনুমোদন করুন
  async approveAffiliateAccount(affiliateId: string) {
    const affiliate = await this.prisma.affiliateAccount.update({
      where: { id: affiliateId },
      data: {
        isApproved: true,
        approvedAt: new Date(),
        status: 'active'
      }
    });

    return { success: true, data: affiliate };
  }

  // অ্যাফিলিয়েট তথ্য পান
  async getAffiliateInfo(affiliateId: string) {
    const affiliate = await this.prisma.affiliateAccount.findUnique({
      where: { id: affiliateId },
      include: {
        user: { select: { name: true, email: true } }
      }
    });

    const stats = await this.prisma.affiliateStats.findUnique({
      where: { affiliateId }
    });

    return {
      success: true,
      data: {
        ...affiliate,
        stats
      }
    };
  }

  private encryptPaymentInfo(info: string): string {
    // Simple encryption - use proper encryption in production
    return Buffer.from(info).toString('base64');
  }
}