import { Injectable } from '@nestjs/common';
import OpenAI from 'openai';

@Injectable()
export class AffiliateAIContentService {
  private openai: OpenAI;

  constructor() {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // AI মার্কেটিং মেসেজ জেনারেট করুন
  async generateMarketingMessage(
    productName: string,
    productDescription: string,
    affiliateLink: string
  ): Promise<string[]> {
    const prompt = `
Generate 3 compelling marketing messages for an affiliate to promote this product:

Product: ${productName}
Description: ${productDescription}
Affiliate Link: ${affiliateLink}

Requirements:
- Each message should be 1-2 sentences
- Include call-to-action
- Make it shareable on social media
- Emphasize benefits, not features

Return as JSON array:
["message1", "message2", "message3"]
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: prompt }],
      temperature: 0.8,
      max_tokens: 300
    });

    try {
      const content = response.choices[0].message.content || '';
      const jsonMatch = content.match(/\[[\s\S]*\]/);
      return jsonMatch ? JSON.parse(jsonMatch[0]) : [];
    } catch (err) {
      return [];
    }
  }

  // AI সোশ্যাল মিডিয়া পোস্ট জেনারেট করুন
  async generateSocialPost(
    productName: string,
    platform: 'instagram' | 'facebook' | 'twitter' | 'tiktok',
    affiliateLink: string
  ): Promise<string> {
    const platformGuidelines: any = {
      instagram: {
        maxLength: 2200,
        tone: 'trendy, visual',
        hashtagCount: 10
      },
      facebook: {
        maxLength: 500,
        tone: 'friendly, engaging',
        hashtagCount: 5
      },
      twitter: {
        maxLength: 280,
        tone: 'punchy, quick',
        hashtagCount: 2
      },
      tiktok: {
        maxLength: 150,
        tone: 'trendy, fun',
        hashtagCount: 5
      }
    };

    const guidelines = platformGuidelines[platform];

    const prompt = `
Generate a viral ${platform} post for this product:
Product: ${productName}
Link: ${affiliateLink}

Guidelines:
- Max ${guidelines.maxLength} characters
- Tone: ${guidelines.tone}
- Include ~${guidelines.hashtagCount} hashtags
- Add call-to-action (click, check, visit, etc)
- Use emojis appropriately

Return ONLY the post text:
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: prompt }],
      temperature: 0.9,
      max_tokens: 300
    });

    return response.choices[0].message.content || '';
  }
}