import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';
import { Cron } from '@nestjs/schedule';

@Injectable()
export class AutoPayoutService {
  private logger = new Logger('AutoPayoutService');

  constructor(private prisma: PrismaService) {}

  // স্বয়ংক্রিয় পেমেন্ট প্রসেসিং (দৈনিক)
  @Cron('0 10 * * *') // প্রতিদিন সকাল 10টায়
  async processAutoPayout() {
    const pendingPayouts = await this.prisma.affiliatePayoutRequest.findMany({
      where: { status: 'pending' },
      include: { affiliate: true }
    });

    for (const payout of pendingPayouts) {
      try {
        const paymentMethod = payout.affiliate.paymentMethod;

        if (paymentMethod === 'bkash') {
          await this.payoutViaBkash(payout);
        } else if (paymentMethod === 'nagad') {
          await this.payoutViaNagad(payout);
        } else if (paymentMethod === 'paypal') {
          await this.payoutViaPayPal(payout);
        } else if (paymentMethod === 'bank_transfer') {
          await this.payoutViaBankTransfer(payout);
        }
      } catch (err) {
        this.logger.error(`Payout failed for request ${payout.id}`, err);
      }
    }
  }

  private async payoutViaBkash(payout: any) {
    try {
      // bKash API কল করুন
      const response = await axios.post(
        'https://api.bkash.com/payout',
        {
          amount: payout.amount,
          accountNumber: this.decryptPaymentInfo(payout.affiliate.paymentAccount),
          transactionId: `AFF-${payout.id}`
        },
        {
          headers: {
            Authorization: `Bearer ${process.env.BKASH_API_KEY}`
          }
        }
      );

      await this.prisma.affiliatePayoutRequest.update({
        where: { id: payout.id },
        data: {
          status: 'completed',
          transactionId: response.data.transactionId,
          processedAt: new Date()
        }
      });

      this.logger.log(`Payout completed via bKash: ${payout.id}`);
    } catch (err) {
      this.logger.error('bKash payout failed:', err);
      throw err;
    }
  }

  private async payoutViaNagad(payout: any) {
    // Nagad API কল
    this.logger.log(`Processing Nagad payout: ${payout.id}`);
  }

  private async payoutViaPayPal(payout: any) {
    try {
      // PayPal API কল করুন
      const response = await axios.post(
        'https://api.paypal.com/v1/payments/payouts',
        {
          sender_batch_header: {
            sender_batch_id: `AFF-${payout.id}`,
            email_subject: 'Affiliate Commission Payout'
          },
          items: [
            {
              recipient_type: 'EMAIL',
              amount: {
                value: payout.amount.toString(),
                currency: 'USD'
              },
              receiver: this.decryptPaymentInfo(payout.affiliate.paymentAccount),
              note: 'Affiliate commission'
            }
          ]
        },
        {
          headers: {
            Authorization: `Bearer ${process.env.PAYPAL_API_KEY}`
          }
        }
      );

      await this.prisma.affiliatePayoutRequest.update({
        where: { id: payout.id },
        data: {
          status: 'completed',
          transactionId: response.data.batch_header.payout_batch_id,
          processedAt: new Date()
        }
      });

      this.logger.log(`Payout completed via PayPal: ${payout.id}`);
    } catch (err) {
      this.logger.error('PayPal payout failed:', err);
      throw err;
    }
  }

  private async payoutViaBankTransfer(payout: any) {
    // ব্যাংক ট্রান্সফার লজ করুন (ম্যানুয়াল প্রসেসিংয়ের জন্য)
    this.logger.log(`Bank transfer payout: ${payout.id}`);
  }

  private decryptPaymentInfo(info: string | null): string {
    if (!info) return '';
    return Buffer.from(info, 'base64').toString('utf-8');
  }
}