import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CommissionService {
  private logger = new Logger('CommissionService');

  constructor(private prisma: PrismaService) {}

  // প্রোডাক্ট-স্পেসিফিক কমিশন সেট করুন
  async setProductCommission(productId: string, commissionPercent: number) {
    // TODO: Create ProductCommission model
    return { success: true };
  }

  // অর্ডার থেকে কমিশন ক্যালকুলেট করুন
  async calculateCommissionFromOrder(orderId: string, affiliateId: string) {
    const order = await this.prisma.order.findUnique({
      where: { id: orderId },
      include: { items: true }
    });

    if (!order) throw new Error('Order not found');

    const affiliate = await this.prisma.affiliateAccount.findUnique({
      where: { id: affiliateId }
    });

    if (!affiliate) throw new Error('Affiliate not found');

    // সব অর্ডার আইটেমের জন্য কমিশন ক্যালকুলেট করুন
    let totalCommission = 0;

    for (const item of order.items) {
      const itemCommission = (Number(item.price) * item.quantity * affiliate.commissionRate) / 100;
      totalCommission += itemCommission;
    }

    // টায়ার 2 এবং 3 কমিশন যোগ করুন (যদি রেফারার থাকে)
    if (affiliate.referrerId) {
      const referrer = await this.prisma.affiliateAccount.findUnique({
        where: { id: affiliate.referrerId }
      });

      if (referrer && referrer.tier2Rate) {
        const tier2Commission = (totalCommission * Number(referrer.tier2Rate)) / 100;
        await this.createCommission(referrer.id, 2, tier2Commission, affiliateId);
      }

      // টায়ার 3
      if (affiliate.referrerChain.length > 1) {
        const tier3AffiliateId = affiliate.referrerChain[1];
        const tier3Affiliate = await this.prisma.affiliateAccount.findUnique({
          where: { id: tier3AffiliateId }
        });

        if (tier3Affiliate && tier3Affiliate.tier3Rate) {
          const tier3Commission = (totalCommission * Number(tier3Affiliate.tier3Rate)) / 100;
          await this.createCommission(tier3Affiliate.id, 3, tier3Commission, affiliateId);
        }
      }
    }

    // টায়ার 1 কমিশন তৈরি করুন
    await this.createCommission(affiliateId, 1, totalCommission, null);

    return { success: true, commission: totalCommission };
  }

  private async createCommission(
    affiliateId: string,
    tier: number,
    amount: number,
    sourceAffiliateId: string | null
  ) {
    const commission = await this.prisma.affiliateCommission.create({
      data: {
        affiliateId,
        tier,
        amount,
        sourceAffiliateId,
        status: 'pending'
      }
    });

    // Stats আপডেট করুন
    await this.prisma.affiliateStats.update({
      where: { affiliateId },
      data: {
        monthlyEarnings: { increment: amount }
      }
    });

    this.logger.log(`Commission created: ${affiliateId} - Tier ${tier} - $${amount}`);

    return commission;
  }
}