import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as crypto from 'crypto';

@Injectable()
export class AffiliateCouponService {
  constructor(private prisma: PrismaService) {}

  // অ্যাফিলিয়েটের জন্য কুপন কোড জেনারেট করুন
  async generateCouponCode(
    affiliateId: string,
    discountType: 'percentage' | 'fixed',
    discountValue: number,
    validDays: number = 30,
    maxUses?: number
  ) {
    const code = `AFF-${affiliateId.slice(0, 5).toUpperCase()}-${crypto.randomBytes(4).toString('hex').toUpperCase()}`;

    const validTo = new Date();
    validTo.setDate(validTo.getDate() + validDays);

    const coupon = await this.prisma.affiliateCoupon.create({
      data: {
        affiliateId,
        code,
        discountType,
        discountValue,
        maxUses,
        validFrom: new Date(),
        validTo
      }
    });

    return {
      success: true,
      data: {
        code: coupon.code,
        discount: `${discountValue}${discountType === 'percentage' ? '%' : '$'}`,
        validUntil: coupon.validTo
      }
    };
  }

  // অ্যাফিলিয়েটের কুপন পান
  async getAffiliateCoupons(affiliateId: string) {
    const coupons = await this.prisma.affiliateCoupon.findMany({
      where: { affiliateId, validTo: { gt: new Date() } }
    });

    return {
      success: true,
      data: coupons.map((c) => ({
        code: c.code,
        discount: `${c.discountValue}${c.discountType === 'percentage' ? '%' : '$'}`,
        uses: `${c.currentUses}${c.maxUses ? '/' + c.maxUses : ''}`,
        validUntil: c.validTo
      }))
    };
  }
}