import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class LeaderboardService {
  constructor(private prisma: PrismaService) {}

  // কমিশন লিডারবোর্ড পান
  async getCommissionLeaderboard(limit = 100) {
    const stats = await this.prisma.affiliateStats.findMany({
      orderBy: { monthlyEarnings: 'desc' },
      take: limit,
      include: {
        affiliate: { select: { user: { select: { name: true } } } }
      }
    });

    return {
      success: true,
      data: stats.map((stat, index) => ({
        rank: index + 1,
        affiliateName: stat.affiliate.user?.name,
        earnings: stat.monthlyEarnings,
        conversions: stat.monthlyConversions
      }))
    };
  }

  // কনভার্সন লিডারবোর্ড পান
  async getConversionLeaderboard(limit = 100) {
    const stats = await this.prisma.affiliateStats.findMany({
      orderBy: { monthlyConversions: 'desc' },
      take: limit,
      include: {
        affiliate: { select: { user: { select: { name: true } } } }
      }
    });

    return {
      success: true,
      data: stats.map((stat, index) => ({
        rank: index + 1,
        affiliateName: stat.affiliate.user?.name,
        conversions: stat.monthlyConversions,
        conversionRate: stat.conversionRate
      }))
    };
  }

  // অ্যাচিভমেন্ট আনলক করুন
  async getAffiliateAchievements(affiliateId: string) {
    const affiliate = await this.prisma.affiliateAccount.findUnique({
      where: { id: affiliateId },
      include: { commissions: true }
    });

    if (!affiliate) throw new Error('Affiliate not found');

    const achievements = [];

    // অর্জন: প্রথম বিক্রয়
    if (affiliate.commissions.length > 0) {
      achievements.push({
        id: 'first_sale',
        name: 'First Sale',
        icon: '🎉',
        unlockedAt: affiliate.commissions[0].createdAt
      });
    }

    // অর্জন: $100 অর্জন
    const total = affiliate.commissions
      .filter((c) => c.status === 'paid')
      .reduce((sum, c) => sum + Number(c.amount), 0);

    if (total >= 100) {
      achievements.push({
        id: 'hundred_dollars',
        name: '$100 Earned',
        icon: '💰',
        progress: (total / 100) * 100
      });
    }

    // অর্জন: 10 রেফারেল
    const referralCount = await this.prisma.affiliateAccount.count({
      where: { referrerId: affiliateId }
    });

    if (referralCount >= 10) {
      achievements.push({
        id: 'ten_referrals',
        name: '10 Referrals',
        icon: '👥',
        progress: (referralCount / 10) * 100
      });
    }

    return {
      success: true,
      data: achievements
    };
  }
}