import { Injectable } from '@nestjs/common';
import * as crypto from 'crypto';
import * as QRCode from 'qrcode';
import * as AWS from 'aws-sdk';

@Injectable()
export class LinkGeneratorService {
  private s3 = new AWS.S3();

  // অ্যাফিলিয়েট লিংক জেনারেট করুন
  async generateAffiliateLink(
    affiliateId: string,
    productId?: string,
    campaignName?: string
  ): Promise<string> {
    // ইউনিক কোড তৈরি করুন
    const linkCode = crypto.randomBytes(8).toString('hex');

    const baseUrl = process.env.APP_URL;
    const link = productId
      ? `${baseUrl}/aff/${affiliateId}/${productId}?code=${linkCode}`
      : `${baseUrl}/aff/${affiliateId}?code=${linkCode}`;

    return link;
  }

  // QR কোড জেনারেট করুন
  async generateQRCode(affiliateLink: string, affiliateId: string): Promise<string> {
    try {
      // QR কোড জেনারেট করুন
      const qrCodeImage = await QRCode.toDataURL(affiliateLink);

      // S3-এ আপলোড করুন
      const key = `affiliate-qr-codes/${affiliateId}/${Date.now()}.png`;

      // Data URL থেকে Buffer তৈরি করুন
      const base64Data = qrCodeImage.replace(/^data:image\/png;base64,/, '');
      const buffer = Buffer.from(base64Data, 'base64');

      await this.s3
        .putObject({
          Bucket: process.env.AWS_S3_BUCKET || '',
          Key: key,
          Body: buffer,
          ContentType: 'image/png'
        })
        .promise();

      return `https://${process.env.AWS_S3_BUCKET}.s3.amazonaws.com/${key}`;
    } catch (err) {
      throw err;
    }
  }

  // মাল্টি ফরম্যাটে শেয়ারেবল লিংক তৈরি করুন
  async generateShareableLinks(affiliateLink: string, affiliateId: string) {
    const qrCodeUrl = await this.generateQRCode(affiliateLink, affiliateId);

    return {
      directLink: affiliateLink,
      qrCodeUrl,
      socialLinks: {
        facebook: `https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(affiliateLink)}`,
        twitter: `https://twitter.com/intent/tweet?url=${encodeURIComponent(affiliateLink)}&text=Check%20this%20out!`,
        whatsapp: `https://wa.me/?text=${encodeURIComponent(affiliateLink)}`,
        telegram: `https://t.me/share/url?url=${encodeURIComponent(affiliateLink)}`
      }
    };
  }
}