import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class MultiLevelAffiliateService {
  private logger = new Logger('MultiLevelAffiliateService');

  constructor(private prisma: PrismaService) {}

  // 3-টায়ার কমিশন কাঠামো সেট করুন
  async setupMultiLevelStructure(
    affiliateId: string,
    input: {
      tier1Rate: number; // Direct sales
      tier2Rate: number; // From tier 1 referrals
      tier3Rate: number; // From tier 2 referrals
    }
  ) {
    const affiliate = await this.prisma.affiliateAccount.update({
      where: { id: affiliateId },
      data: {
        commissionRate: input.tier1Rate,
        tier2Rate: input.tier2Rate,
        tier3Rate: input.tier3Rate
      }
    });

    return { success: true, data: affiliate };
  }

  // টায়ার পারফরম্যান্স পান
  async getTierPerformance(affiliateId: string) {
    const affiliate = await this.prisma.affiliateAccount.findUnique({
      where: { id: affiliateId },
      include: {
        commissions: {
          where: { status: 'approved' }
        }
      }
    });

    if (!affiliate) throw new Error('Affiliate not found');

    // টায়ার-ওয়াইজ আর্নিং
    const tier1Earnings = affiliate.commissions
      .filter((c) => c.tier === 1)
      .reduce((sum, c) => sum + Number(c.amount), 0);

    const tier2Earnings = affiliate.commissions
      .filter((c) => c.tier === 2)
      .reduce((sum, c) => sum + Number(c.amount), 0);

    const tier3Earnings = affiliate.commissions
      .filter((c) => c.tier === 3)
      .reduce((sum, c) => sum + Number(c.amount), 0);

    // রেফারেল কাউন্ট
    const tier1Referrals = await this.prisma.affiliateAccount.count({
      where: { referrerId: affiliateId }
    });

    const tier2Referrals = await this.prisma.affiliateAccount.count({
      where: { referrerChain: { has: affiliateId } }
    });

    return {
      success: true,
      data: {
        tier1: { earnings: tier1Earnings, referrals: tier1Referrals },
        tier2: { earnings: tier2Earnings, referrals: tier2Referrals },
        tier3: { earnings: tier3Earnings },
        totalEarnings: tier1Earnings + tier2Earnings + tier3Earnings
      }
    };
  }

  // নৈতিক MLM চেক (সীমা নিশ্চিত করুন)
  async validateEthicalMLM(affiliateId: string) {
    const affiliate = await this.prisma.affiliateAccount.findUnique({
      where: { id: affiliateId }
    });

    if (!affiliate) throw new Error('Affiliate not found');

    // নিয়ম: কমিশনের মাধ্যমে পণ্য বিক্রয় হওয়া উচিত, শুধু রেফারেল নয়
    const conversions = await this.prisma.affiliateConversion.count({
      where: { affiliateId }
    });

    const referrals = await this.prisma.affiliateAccount.count({
      where: { referrerId: affiliateId }
    });

    // প্রকৃত বিক্রয়ের চেয়ে বেশি রেফারেল থাকলে সতর্ক করুন
    if (conversions === 0 && referrals > 10) {
      this.logger.warn(`Potential unethical MLM activity: ${affiliateId}`);
      return {
        isEthical: false,
        warning: 'No actual sales, only referrals'
      };
    }

    return { isEthical: true };
  }
}