import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as nodemailer from 'nodemailer';

@Injectable()
export class AffiliateNotificationService {
  private logger = new Logger('AffiliateNotificationService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // নতুন বিক্রি নোটিফিকেশন
  async notifyNewSale(affiliateId: string, saleAmount: number, commission: number) {
    const affiliate = await this.prisma.affiliateAccount.findUnique({
      where: { id: affiliateId },
      include: { user: true }
    });

    if (!affiliate || !affiliate.user.email) return;

    const emailContent = `
Dear ${affiliate.user.name},

Congratulations! You have made a new sale!

Sale Amount: $${saleAmount}
Your Commission: $${commission}

Your pending balance: Check your dashboard for details.

Keep promoting and earning!

Best regards,
The Affiliate Team
    `;

    await this.mailer.sendMail({
      from: 'noreply@affiliate-platform.com',
      to: affiliate.user.email,
      subject: `🎉 New Sale! You earned $${commission}`,
      text: emailContent
    });

    this.logger.log(`Sale notification sent to ${affiliate.id}`);
  }

  // পেমেন্ট প্রসেসড নোটিফিকেশন
  async notifyPayoutProcessed(payoutId: string) {
    const payout = await this.prisma.affiliatePayoutRequest.findUnique({
      where: { id: payoutId },
      include: { affiliate: { include: { user: true } } }
    });

    if (!payout || !payout.affiliate.user.email) return;

    const emailContent = `
Dear ${payout.affiliate.user.name},

Your affiliate payout has been processed!

Amount: $${payout.amount}
Payment Method: ${payout.affiliate.paymentMethod}
Transaction ID: ${payout.transactionId}

The funds should reach your account within 1-3 business days.

Thank you for being part of our affiliate program!

Best regards,
The Affiliate Team
    `;

    await this.mailer.sendMail({
      from: 'noreply@affiliate-platform.com',
      to: payout.affiliate.user.email,
      subject: `💰 Payout Processed - $${payout.amount}`,
      text: emailContent
    });

    this.logger.log(`Payout notification sent for ${payoutId}`);
  }
}