import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as crypto from 'crypto';

@Injectable()
export class ProductAffiliateLinksService {
  private logger = new Logger('ProductAffiliateLinksService');

  constructor(private prisma: PrismaService) {}

  // প্রোডাক্টের জন্য অ্যাফিলিয়েট লিংক তৈরি করুন
  async generateProductLink(
    affiliateId: string,
    productId: string
  ): Promise<string> {
    try {
      // বিদ্যমান লিংক চেক করুন
      const existing = await this.prisma.productAffiliateLink.findUnique({
        where: {
          productId_affiliateId: { productId, affiliateId }
        }
      });

      if (existing) {
        return existing.fullUrl;
      }

      // নতুন লিংক তৈরি করুন
      const linkCode = crypto.randomBytes(12).toString('hex');
      const fullUrl = `${process.env.APP_URL}/aff/${affiliateId}/${productId}/${linkCode}`;

      const link = await this.prisma.productAffiliateLink.create({
        data: {
          productId,
          affiliateId,
          linkCode,
          fullUrl
        }
      });

      this.logger.log(`Product link created: ${productId} for affiliate ${affiliateId}`);

      return fullUrl;
    } catch (err) {
      this.logger.error('Link generation failed:', err);
      throw err;
    }
  }

  // সব প্রোডাক্টের জন্য লিংক পান
  async getAffiliateProductLinks(affiliateId: string) {
    const links = await this.prisma.productAffiliateLink.findMany({
      where: { affiliateId },
      include: { product: { select: { title: true, price: true } } }
    });

    return {
      success: true,
      data: links.map((link) => ({
        productId: link.productId,
        productName: link.product.title,
        productPrice: link.product.price,
        affiliateLink: link.fullUrl,
        clicks: link.clicks,
        conversions: link.conversions
      }))
    };
  }

  // রিয়েল-টাইম লিংক পারফরম্যান্স
  async getLinkPerformance(linkCode: string) {
    const link = await this.prisma.productAffiliateLink.findUnique({
      where: { linkCode }
    });

    if (!link) throw new Error('Link not found');

    const conversionRate = link.clicks > 0 ? ((link.conversions / link.clicks) * 100).toFixed(2) : '0';

    return {
      success: true,
      data: {
        clicks: link.clicks,
        conversions: link.conversions,
        conversionRate: parseFloat(conversionRate as string)
      }
    };
  }
}