import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as crypto from 'crypto';

@Injectable()
export class TrackingPixelService {
  private logger = new Logger('TrackingPixelService');

  constructor(private prisma: PrismaService) {}

  // ট্র্যাকিং পিক্সেল ইমেজ জেনারেট করুন (1x1 স্বচ্ছ PNG)
  async generateTrackingPixel(affiliateId: string, linkCode: string): Promise<Buffer> {
    // 1x1 স্বচ্ছ PNG
    const pixelBuffer = Buffer.from([
      0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a, 0x00, 0x00, 0x00, 0x0d, 0x49, 0x48, 0x44, 0x52,
      0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x08, 0x06, 0x00, 0x00, 0x00, 0x1f, 0x15, 0xc4,
      0x89, 0x00, 0x00, 0x00, 0x0d, 0x49, 0x44, 0x41, 0x54, 0x08, 0x5b, 0x63, 0xf8, 0xcf, 0xc0, 0x00,
      0x00, 0x03, 0x01, 0x01, 0x00, 0x18, 0xdd, 0x8d, 0xb4, 0x00, 0x00, 0x00, 0x00, 0x49, 0x45, 0x4e,
      0x44, 0xae, 0x42, 0x60, 0x82
    ]);

    return pixelBuffer;
  }

  // ক্লিক ট্র্যাক করুন
  async trackClick(
    affiliateId: string,
    linkCode: string,
    ipAddress: string,
    userAgent: string,
    referrer?: string
  ) {
    const event = await this.prisma.trackingPixelEvent.create({
      data: {
        affiliateId,
        linkCode,
        eventType: 'click',
        ipAddress,
        userAgent,
        referrer
      }
    });

    // লিংক ক্লিক কাউন্ট আপডেট করুন
    await this.prisma.productAffiliateLink.updateMany({
      where: { linkCode },
      data: { clicks: { increment: 1 } }
    });

    // অ্যাফিলিয়েট স্ট্যাটস আপডেট করুন
    await this.prisma.affiliateStats.update({
      where: { affiliateId },
      data: { monthlyClicks: { increment: 1 } }
    });

    return event.id;
  }

  // কুকি-ভিত্তিক সেশন তৈরি করুন (30 দিন)
  async createAffiliateCookie(
    affiliateId: string,
    ipAddress: string,
    userAgent: string
  ): Promise<string> {
    const cookieToken = crypto.randomBytes(32).toString('hex');
    const expiresAt = new Date(Date.now() + 30 * 24 * 60 * 60 * 1000); // 30 দিন

    await this.prisma.affiliateCookieSession.create({
      data: {
        affiliateId,
        cookieToken,
        ipAddress,
        userAgent,
        expiresAt
      }
    });

    return cookieToken;
  }

  // কুকি ভেরিফাই করুন
  async verifyCookie(cookieToken: string): Promise<string | null> {
    const session = await this.prisma.affiliateCookieSession.findUnique({
      where: { cookieToken }
    });

    if (!session || new Date() > session.expiresAt) {
      return null;
    }

    return session.affiliateId;
  }

  // কনভার্সন ট্র্যাক করুন (কুকি থেকে)
  async trackConversionWithCookie(cookieToken: string, orderId: string) {
    const affiliateId = await this.verifyCookie(cookieToken);

    if (!affiliateId) {
      this.logger.warn('Invalid or expired cookie for conversion');
      return null;
    }

    // কনভার্সন ট্র্যাক করুন
    const conversion = await this.prisma.affiliateConversion.create({
      data: {
        affiliateId,
        orderId,
        saleAmount: 0 // Will be set from order
      }
    });

    return conversion;
  }
}