import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class AffiliateTrackingService {
  private logger = new Logger('AffiliateTrackingService');

  constructor(private prisma: PrismaService) {}

  // লিংক ক্লিক ট্র্যাক করুন
  async trackClick(
    affiliateId: string,
    linkCode: string,
    productId: string | null,
    ipAddress: string,
    userAgent: string
  ) {
    const click = await this.prisma.affiliateClick.create({
      data: {
        affiliateId,
        productId,
        linkCode,
        ipAddress,
        userAgent
      }
    });

    // Stats আপডেট করুন
    await this.prisma.affiliateStats.update({
      where: { affiliateId },
      data: {
        monthlyClicks: { increment: 1 }
      }
    });

    this.logger.log(`Affiliate click tracked: ${affiliateId}`);

    return { success: true, clickId: click.id };
  }

  // কনভার্সন ট্র্যাক করুন (অর্ডার তৈরির সময়)
  async trackConversion(
    affiliateId: string,
    orderId: string,
    productId: string | null,
    saleAmount: number
  ) {
    const conversion = await this.prisma.affiliateConversion.create({
      data: {
        affiliateId,
        orderId,
        productId,
        saleAmount
      }
    });

    // ক্লিক অবজেক্ট আপডেট করুন (যদি থাকে)
    await this.prisma.affiliateClick.updateMany({
      where: { affiliateId },
      data: { converted: true }
    });

    // Stats আপডেট করুন
    const stats = await this.prisma.affiliateStats.findUnique({
      where: { affiliateId }
    });

    const conversionRate = stats
      ? (((stats.monthlyConversions + 1) / (stats.monthlyClicks || 1)) * 100).toFixed(2)
      : '0';

    await this.prisma.affiliateStats.update({
      where: { affiliateId },
      data: {
        monthlyConversions: { increment: 1 },
        conversionRate: parseFloat(conversionRate as string)
      }
    });

    this.logger.log(`Affiliate conversion tracked: ${affiliateId} - Order ${orderId}`);

    return { success: true, conversionId: conversion.id };
  }

  // কনভার্সন রেট পান
  async getConversionMetrics(affiliateId: string) {
    const clicks = await this.prisma.affiliateClick.count({
      where: { affiliateId }
    });

    const conversions = await this.prisma.affiliateConversion.count({
      where: { affiliateId }
    });

    const conversionRate = clicks > 0 ? ((conversions / clicks) * 100).toFixed(2) : '0';

    return {
      success: true,
      data: {
        totalClicks: clicks,
        totalConversions: conversions,
        conversionRate: parseFloat(conversionRate as string)
      }
    };
  }
}