import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class AffiliateWalletService {
  private logger = new Logger('AffiliateWalletService');

  constructor(private prisma: PrismaService) {}

  // ওয়ালেট ব্যালেন্স পান
  async getWalletBalance(affiliateId: string) {
    const commissions = await this.prisma.affiliateCommission.findMany({
      where: { affiliateId }
    });

    const pending = commissions
      .filter((c) => c.status === 'pending')
      .reduce((sum, c) => sum + Number(c.amount), 0);

    const approved = commissions
      .filter((c) => c.status === 'approved')
      .reduce((sum, c) => sum + Number(c.amount), 0);

    const paid = commissions
      .filter((c) => c.status === 'paid')
      .reduce((sum, c) => sum + Number(c.amount), 0);

    return {
      success: true,
      data: {
        pendingBalance: pending,
        availableBalance: approved,
        totalPaidOut: paid,
        totalEarnings: pending + approved + paid
      }
    };
  }

  // পেমেন্ট হিস্ট্রি পান
  async getPaymentHistory(affiliateId: string, limit = 50) {
    const payouts = await this.prisma.affiliatePayoutRequest.findMany({
      where: { affiliateId },
      orderBy: { requestedAt: 'desc' },
      take: limit
    });

    return {
      success: true,
      data: payouts
    };
  }

  // পেমেন্ট রিকোয়েস্ট করুন
  async requestPayout(
    affiliateId: string,
    amount: number,
    paymentMethod: string
  ) {
    const affiliate = await this.prisma.affiliateAccount.findUnique({
      where: { id: affiliateId }
    });

    if (!affiliate) throw new Error('Affiliate not found');

    // ব্যালেন্স চেক করুন
    const balance = await this.getWalletBalance(affiliateId);

    if (balance.data.availableBalance < amount) {
      throw new Error('Insufficient balance');
    }

    // পেমেন্ট রিকোয়েস্ট তৈরি করুন
    const payout = await this.prisma.affiliatePayoutRequest.create({
      data: {
        affiliateId,
        amount
      }
    });

    // পেন্ডিং কমিশন অ্যাপ্রুভ করুন
    await this.prisma.affiliateCommission.updateMany({
      where: {
        affiliateId,
        status: 'pending'
      },
      data: { status: 'approved' }
    });

    this.logger.log(`Payout requested: ${affiliateId} - $${amount}`);

    return { success: true, payoutId: payout.id };
  }
}