import { Injectable, BadRequestException, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';
import * as Pinecone from '@pinecone-database/pinecone';

@Injectable()
export class AiService {
  private logger = new Logger('AiService');
  private openai: OpenAI;
  private pinecone: any;

  constructor(private prisma: PrismaService) {
    this.initializeAi();
  }

  private initializeAi() {
    this.openai = new OpenAI({
      apiKey: process.env.OPENAI_API_KEY
    });

    // Initialize Pinecone for vector embeddings
    if (process.env.PINECONE_API_KEY) {
      this.pinecone = new Pinecone.Pinecone({
        apiKey: process.env.PINECONE_API_KEY,
        environment: process.env.PINECONE_ENVIRONMENT
      });
    }
  }

  // ===== PRODUCT DESCRIPTION GENERATOR =====
  async generateProductDescription(
    tenantId: string,
    productData: {
      title: string;
      category: string;
      price: number;
      niche?: string;
      features?: string[];
      targetAudience?: string;
    }
  ) {
    try {
      const prompt = `
You are an expert e-commerce copywriter. Generate a compelling product description for:
- Title: ${productData.title}
- Category: ${productData.category}
- Price: $${productData.price}
${productData.niche ? `- Niche: ${productData.niche}` : ''}
${productData.features ? `- Key Features: ${productData.features.join(', ')}` : ''}
${productData.targetAudience ? `- Target Audience: ${productData.targetAudience}` : ''}

Requirements:
1. 3-4 sentences maximum
2. Emphasize benefits, not just features
3. Include call-to-action
4. SEO-friendly
5. Professional but friendly tone

Generate ONLY the description, no additional text.`;

      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: prompt }],
        temperature: 0.7,
        max_tokens: 300
      });

      const description = response.choices[0].message.content;

      return {
        success: true,
        data: {
          description,
          usage: {
            tokens: response.usage?.total_tokens,
            model: 'gpt-3.5-turbo'
          }
        }
      };
    } catch (err: any) {
      this.logger.error('Failed to generate description:', err);
      throw new BadRequestException('AI generation failed');
    }
  }

  // ===== SEO METADATA GENERATOR =====
  async generateSeoMetadata(
    tenantId: string,
    productData: {
      title: string;
      description: string;
      category: string;
      price: number;
    }
  ) {
    try {
      const prompt = `
Generate SEO metadata for an e-commerce product:
- Title: ${productData.title}
- Description: ${productData.description}
- Category: ${productData.category}

Return ONLY valid JSON with no markdown formatting:
{
  "seoTitle": "SEO optimized title (60 chars max)",
  "seoDescription": "Meta description (160 chars max)",
  "seoKeywords": "comma,separated,keywords,max,5",
  "schemaCategory": "appropriate schema.org category"
}`;

      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: prompt }],
        temperature: 0.5,
        max_tokens: 200
      });

      const text = response.choices[0].message.content || '';
      const jsonMatch = text.match(/\{[\s\S]*\}/);
      const metadata = jsonMatch ? JSON.parse(jsonMatch[0]) : {};

      return {
        success: true,
        data: metadata
      };
    } catch (err: any) {
      this.logger.error('Failed to generate SEO metadata:', err);
      throw new BadRequestException('SEO generation failed');
    }
  }

  // ===== PRICING SUGGESTION ENGINE =====
  async suggestPricing(
    tenantId: string,
    productData: {
      title: string;
      category: string;
      costPrice: number;
      competition?: {
        name: string;
        price: number;
      }[];
      profitMargin?: number; // default 40%
    }
  ) {
    try {
      const competitionText = productData.competition
        ? `Competitors: ${productData.competition.map((c) => `${c.name} ($${c.price})`).join(', ')}`
        : '';

      const prompt = `
You are a pricing strategist. Suggest optimal pricing for:
- Product: ${productData.title}
- Category: ${productData.category}
- Cost Price: $${productData.costPrice}
- Desired Margin: ${productData.profitMargin || 40}%
${competitionText}

Return ONLY JSON with no markdown:
{
  "recommendedPrice": number,
  "minPrice": number,
  "maxPrice": number,
  "strategy": "penetration|value|premium",
  "reasoning": "brief explanation",
  "profitMargin": number
}`;

      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: prompt }],
        temperature: 0.6,
        max_tokens: 250
      });

      const text = response.choices[0].message.content || '';
      const jsonMatch = text.match(/\{[\s\S]*\}/);
      const pricing = jsonMatch ? JSON.parse(jsonMatch[0]) : {};

      return {
        success: true,
        data: pricing
      };
    } catch (err: any) {
      this.logger.error('Failed to suggest pricing:', err);
      throw new BadRequestException('Pricing suggestion failed');
    }
  }

  // ===== SOCIAL MEDIA POST GENERATOR =====
  async generateSocialPost(
    tenantId: string,
    productData: {
      title: string;
      description: string;
      platform: 'instagram' | 'facebook' | 'tiktok' | 'twitter';
      hashtags?: boolean;
    }
  ) {
    try {
      const platformGuidelines: any = {
        instagram: '150 chars, trendy, hashtags',
        facebook: '300 chars, friendly, emoji-friendly',
        tiktok: '150 chars, casual, trendy',
        twitter: '280 chars max, punchy'
      };

      const prompt = `
Write a social media post for ${productData.platform} about this product:
- Title: ${productData.title}
- Description: ${productData.description}

Guidelines: ${platformGuidelines[productData.platform]}
${productData.hashtags ? '- Include relevant hashtags' : ''}

Generate ONLY the post text, no explanations.`;

      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: prompt }],
        temperature: 0.8,
        max_tokens: 300
      });

      return {
        success: true,
        data: {
          post: response.choices[0].message.content,
          platform: productData.platform
        }
      };
    } catch (err: any) {
      throw new BadRequestException('Social post generation failed');
    }
  }

  // ===== AI CHATBOT (Trained on store data) =====
  async chatbotResponse(
    tenantId: string,
    message: string,
    conversationHistory?: Array<{ role: string; content: string }>
  ) {
    try {
      // Get store context
      const store = await this.prisma.tenantSettings.findUnique({
        where: { tenantId }
      });

      const systemPrompt = `
You are a helpful customer service assistant for ${store?.storeName || 'our store'}.
- Be friendly and professional
- Answer questions about products, orders, shipping
- Offer help with returns and refunds
- Keep responses concise (max 150 words)
- If you don't know, suggest contacting support at ${store?.storeEmail}`;

      const messages = [
        { role: 'system', content: systemPrompt },
        ...(conversationHistory || []),
        { role: 'user', content: message }
      ];

      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: messages as any,
        temperature: 0.7,
        max_tokens: 200
      });

      return {
        success: true,
        data: {
          reply: response.choices[0].message.content,
          conversationId: `conv_${Date.now()}`
        }
      };
    } catch (err: any) {
      throw new BadRequestException('Chatbot failed');
    }
  }

  // ===== TREND DETECTOR (AI analyzes bestsellers) =====
  async detectTrends(tenantId: string) {
    try {
      const topProducts = await this.prisma.product.findMany({
        where: { tenantId },
        include: { _count: { select: { orders: true } } },
        orderBy: { createdAt: 'desc' },
        take: 50
      });

      const prompt = `
Analyze these bestselling products and identify trends:
${topProducts
  .slice(0, 10)
  .map((p) => `- ${p.title} (${p._count.orders} orders)`)
  .join('\n')}

Identify:
1. Emerging categories
2. Price point trends
3. Customer preferences
4. Seasonal opportunities

Return JSON with no markdown:
{
  "hotCategories": ["category1", "category2"],
  "pricePointTrend": "budget|midrange|premium",
  "customerPreference": "description",
  "recommendations": ["rec1", "rec2"]
}`;

      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: prompt }],
        temperature: 0.6,
        max_tokens: 300
      });

      const text = response.choices[0].message.content || '';
      const jsonMatch = text.match(/\{[\s\S]*\}/);
      const trends = jsonMatch ? JSON.parse(jsonMatch[0]) : {};

      return {
        success: true,
        data: trends
      };
    } catch (err: any) {
      throw new BadRequestException('Trend detection failed');
    }
  }

  // ===== INVENTORY PREDICTOR =====
  async predictInventory(tenantId: string, productId: string) {
    try {
      const orders = await this.prisma.order.findMany({
        where: {
          tenantId,
          items: { some: { productId } },
          status: { not: 'CANCELLED' }
        },
        select: { createdAt: true }
      });

      if (orders.length < 5) {
        return {
          success: true,
          data: {
            prediction: 'Insufficient data',
            confidence: 0,
            recommendedStock: 10
          }
        };
      }

      // Simple time-series analysis
      const dates = orders.map((o) => new Date(o.createdAt).getTime());
      const intervals = [];
      for (let i = 1; i < dates.length; i++) {
        intervals.push((dates[i] - dates[i - 1]) / (1000 * 60 * 60 * 24)); // days
      }

      const avgDaysBetweenOrders = intervals.reduce((a, b) => a + b, 0) / intervals.length;
      const ordersPerMonth = 30 / avgDaysBetweenOrders;

      return {
        success: true,
        data: {
          prediction: `~${Math.round(ordersPerMonth)} orders/month`,
          confidence: Math.min(orders.length / 20, 1),
          recommendedStock: Math.ceil(ordersPerMonth * 2),
          reorderPoint: Math.ceil(ordersPerMonth * 0.5)
        }
      };
    } catch (err: any) {
      throw new BadRequestException('Prediction failed');
    }
  }
}