import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class AiAnalyticsService {
  private logger = new Logger('AiAnalyticsService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // ট্রেন্ড ডিটেক্ট করুন
  async detectTrends(tenantId: string) {
    try {
      // সর্বোচ্চ বিক্রয়ের পণ্য পান
      const topProducts = await this.prisma.product.findMany({
        where: { tenantId },
        include: { _count: { select: { orders: true } } },
        orderBy: { createdAt: 'desc' },
        take: 20
      });

      const prompt = `
Analyze these bestselling products and identify market trends:
${topProducts
  .slice(0, 10)
  .map((p) => `- ${p.title} (${p._count.orders} orders)`)
  .join('\n')}

Identify:
1. Emerging product categories
2. Price point trends
3. Customer preferences
4. Seasonal opportunities

Return JSON:
{
  "hotCategories": ["cat1", "cat2"],
  "pricePointTrend": "budget|midrange|premium",
  "predictions": ["prediction1"],
  "recommendations": ["action1"]
}
`;

      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: prompt }],
        temperature: 0.7,
        max_tokens: 400
      });

      const jsonMatch = response.choices[0].message.content?.match(/\{[\s\S]*\}/);
      const trends = jsonMatch ? JSON.parse(jsonMatch[0]) : {};

      return { success: true, data: trends };
    } catch (err) {
      this.logger.error('Trend detection failed:', err);
      throw err;
    }
  }

  // স্টক আউট প্রেডিক্ট করুন
  async predictInventory(tenantId: string, productId: string) {
    try {
      // অর্ডার হিস্ট্রি পান
      const orders = await this.prisma.order.findMany({
        where: {
          tenantId,
          items: { some: { productId } },
          status: { not: 'CANCELLED' }
        },
        select: { createdAt: true },
        orderBy: { createdAt: 'asc' }
      });

      if (orders.length < 5) {
        return {
          prediction: 'Insufficient data',
          confidence: 0,
          recommendedStock: 10
        };
      }

      // Time-series analysis করুন
      const dates = orders.map((o) => new Date(o.createdAt).getTime());
      const intervals: number[] = [];

      for (let i = 1; i < dates.length; i++) {
        intervals.push((dates[i] - dates[i - 1]) / (1000 * 60 * 60 * 24)); // Days
      }

      const avgDaysBetweenOrders =
        intervals.reduce((a, b) => a + b, 0) / intervals.length;
      const ordersPerMonth = 30 / avgDaysBetweenOrders;

      // প্রেডিকশন
      const product = await this.prisma.product.findUnique({
        where: { id: productId }
      });

      const stockoutDate = new Date();
      stockoutDate.setDate(
        stockoutDate.getDate() +
          (product?.stock || 0) / ordersPerMonth
      );

      // সেভ করুন
      await this.prisma.inventoryForecast.upsert({
        where: { tenantId_productId: { tenantId, productId } },
        create: {
          tenantId,
          productId,
          currentStock: product?.stock || 0,
          dailyConsumption: ordersPerMonth / 30,
          forecastedStockout: stockoutDate,
          recommendedReorder: Math.ceil(ordersPerMonth * 2),
          confidence: Math.min(orders.length / 50, 1)
        },
        update: {
          currentStock: product?.stock || 0,
          dailyConsumption: ordersPerMonth / 30,
          forecastedStockout: stockoutDate,
          recommendedReorder: Math.ceil(ordersPerMonth * 2),
          confidence: Math.min(orders.length / 50, 1)
        }
      });

      return {
        prediction: `${Math.round(ordersPerMonth)}/month`,
        confidence: Math.min(orders.length / 50, 1),
        recommendedStock: Math.ceil(ordersPerMonth * 2),
        stockoutWarning: stockoutDate < new Date(Date.now() + 7 * 24 * 60 * 60 * 1000)
      };
    } catch (err) {
      this.logger.error('Inventory prediction failed:', err);
      throw err;
    }
  }
}