import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class ChatbotService {
  private logger = new Logger('ChatbotService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // চ্যাটবট রেসপন্স পান
  async getChatResponse(
    tenantId: string,
    visitorId: string,
    message: string
  ): Promise<string> {
    try {
      // ১. কথোপকথনের ইতিহাস পান
      const conversation = await this.getOrCreateConversation(tenantId, visitorId);

      // २. স্টোর ডেটা পান (প্রশিক্ষণের জন্য)
      const storeContext = await this.getStoreContext(tenantId);

      // ३. সিস্টেম প্রম্পট তৈরি করুন
      const systemPrompt = `
You are a helpful customer service chatbot for "${storeContext.storeName}".

Store Information:
- Products: ${storeContext.products.slice(0, 5).map((p: any) => p.title).join(', ')}
- Shipping: Free over $${storeContext.freeShippingThreshold}
- Return Policy: 30 days

FAQs:
${storeContext.faqs.map((faq: any) => `Q: ${faq.question}\nA: ${faq.answer}`).join('\n\n')}

Be helpful, friendly, and professional. If asked about something not in your knowledge, offer to escalate to human support.
`;

      // ४. OpenAI কল করুন
      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [
          { role: 'system', content: systemPrompt },
          ...conversation.messages.map((msg) => ({
            role: msg.sender as 'user' | 'assistant',
            content: msg.message
          })),
          { role: 'user', content: message }
        ],
        temperature: 0.7,
        max_tokens: 200
      });

      const botResponse = response.choices[0].message.content || '';

      // ५. কথোপকথন সংরক্ষণ করুন
      await this.saveMessages(conversation.id, message, botResponse);

      return botResponse;
    } catch (err) {
      this.logger.error('Chatbot response failed:', err);
      throw err;
    }
  }

  private async getOrCreateConversation(tenantId: string, visitorId: string) {
    let conversation = await this.prisma.chatConversation.findFirst({
      where: { tenantId, visitorId, status: 'active' },
      include: { messages: { orderBy: { createdAt: 'asc' }, take: 10 } }
    });

    if (!conversation) {
      conversation = await this.prisma.chatConversation.create({
        data: { tenantId, visitorId, status: 'active' },
        include: { messages: true }
      });
    }

    return conversation;
  }

  private async getStoreContext(tenantId: string) {
    const products = await this.prisma.product.findMany({
      where: { tenantId },
      select: { title: true, price: true },
      take: 10
    });

    const settings = await this.prisma.tenantSettings.findUnique({
      where: { tenantId }
    });

    // TODO: Get FAQs from a FAQ table
    const faqs = [
      { question: 'What is your return policy?', answer: '30-day money-back guarantee' },
      { question: 'Do you offer free shipping?', answer: `Yes, free shipping on orders over $${settings?.shippingFreeAfter || 100}` }
    ];

    return {
      storeName: settings?.storeName || 'Store',
      products,
      freeShippingThreshold: settings?.shippingFreeAfter || 100,
      faqs
    };
  }

  private async saveMessages(conversationId: string, userMessage: string, botMessage: string) {
    await Promise.all([
      this.prisma.chatMessage.create({
        data: { conversationId, sender: 'user', message: userMessage }
      }),
      this.prisma.chatMessage.create({
        data: { conversationId, sender: 'bot', message: botMessage, confidence: 0.9 }
      })
    ]);
  }
}