import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';
import * as AWS from 'aws-sdk';

@Injectable()
export class ImageProcessorService {
  private logger = new Logger('ImageProcessorService');
  private s3 = new AWS.S3();

  constructor(private prisma: PrismaService) {}

  // ব্যাকগ্রাউন্ড রিমুভ করুন
  async removeBackground(tenantId: string, imageUrl: string): Promise<string> {
    try {
      this.logger.log(`Processing image: ${imageUrl}`);

      // ১. Remove.bg API ব্যবহার করুন (বা অন্য সেবা)
      const processedUrl = await this.callRemoveBgApi(imageUrl);

      // २. S3-এ সংরক্ষণ করুন
      const s3Url = await this.uploadToS3(tenantId, processedUrl);

      // ३. ডাটাবেস রেকর্ড সংরক্ষণ করুন
      await this.prisma.processedImage.create({
        data: {
          tenantId,
          originalUrl: imageUrl,
          processedUrl: s3Url,
          processedFormat: 'png',
          status: 'completed'
        }
      });

      return s3Url;
    } catch (err) {
      this.logger.error('Image processing failed:', err);
      
      // ফেইল স্টেটাস সংরক্ষণ করুন
      await this.prisma.processedImage.create({
        data: {
          tenantId,
          originalUrl: imageUrl,
          status: 'failed',
          error: (err as Error).message
        }
      });

      throw err;
    }
  }

  private async callRemoveBgApi(imageUrl: string): Promise<string> {
    // Remove.bg API ব্যবহার করুন
    const response = await axios.post(
      'https://api.remove.bg/v1.0/removebg',
      { image_url: imageUrl, format: 'PNG' },
      {
        headers: {
          'X-Api-Key': process.env.REMOVE_BG_API_KEY
        },
        responseType: 'arraybuffer'
      }
    );

    return response.data;
  }

  private async uploadToS3(tenantId: string, imageData: Buffer): Promise<string> {
    const key = `processed-images/${tenantId}/${Date.now()}.png`;

    await this.s3
      .putObject({
        Bucket: process.env.AWS_S3_BUCKET,
        Key: key,
        Body: imageData,
        ContentType: 'image/png'
      })
      .promise();

    return `https://${process.env.AWS_S3_BUCKET}.s3.amazonaws.com/${key}`;
  }
}