import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class MarketingCopyService {
  private logger = new Logger('MarketingCopyService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // সোশ্যাল মিডিয়া পোস্ট জেনারেট করুন
  async generateSocialPost(tenantId: string, input: {
    productTitle: string;
    description: string;
    price: number;
    platform: 'instagram' | 'facebook' | 'twitter' | 'tiktok';
  }) {
    const platformGuidelines: any = {
      instagram: {
        maxLength: 2200,
        tone: 'trendy, visual',
        hashtagCount: 10,
        emoji: true
      },
      facebook: {
        maxLength: 500,
        tone: 'friendly, casual',
        hashtagCount: 5,
        emoji: true
      },
      twitter: {
        maxLength: 280,
        tone: 'punchy, engaging',
        hashtagCount: 2,
        emoji: false
      },
      tiktok: {
        maxLength: 150,
        tone: 'trendy, fun',
        hashtagCount: 5,
        emoji: true
      }
    };

    const guidelines = platformGuidelines[input.platform];

    const prompt = `
Generate a viral ${input.platform} post for:
Product: ${input.productTitle}
Description: ${input.description}
Price: $${input.price}

Platform guidelines:
- Max ${guidelines.maxLength} characters
- Tone: ${guidelines.tone}
- Include ~${guidelines.hashtagCount} hashtags
${guidelines.emoji ? '- Use emojis' : '- No emojis'}

Make it engaging and click-worthy!

Return ONLY the post text.
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: prompt }],
      temperature: 0.9,
      max_tokens: 300
    });

    const post = response.choices[0].message.content?.trim() || '';

    // সেভ করুন
    await this.prisma.aiGeneratedContent.create({
      data: {
        tenantId,
        type: 'social_post',
        content: post,
        model: 'gpt-3.5-turbo',
        tokens: 200
      }
    });

    return { success: true, data: { post, platform: input.platform } };
  }

  // ইমেইল সাবজেক্ট লাইন জেনারেট করুন
  async generateEmailSubjectLine(tenantId: string, input: {
    campaignType: 'promotional' | 'newsletter' | 'transactional';
    content: string;
  }) {
    const prompt = `
Generate 5 compelling email subject lines for a ${input.campaignType} email:
Content: ${input.content}

Requirements:
- 50-60 characters max
- High open rate potential
- Personalization friendly

Return as JSON array: ["subject1", "subject2", ...]
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: prompt }],
      temperature: 0.8,
      max_tokens: 200
    });

    const jsonMatch = response.choices[0].message.content?.match(/\[[\s\S]*\]/);
    const subjects = jsonMatch ? JSON.parse(jsonMatch[0]) : [];

    return { success: true, data: { subjects } };
  }
}