import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class PricingService {
  private logger = new Logger('PricingService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // প্রাইস সাজেশন পান
  async suggestPrice(tenantId: string, input: {
    productTitle: string;
    category: string;
    costPrice: number;
    competitors?: Array<{ name: string; price: number }>;
  }) {
    try {
      const prompt = `
Analyze pricing for product: "${input.productTitle}"
Category: ${input.category}
Cost Price: $${input.costPrice}

Competitors:
${input.competitors?.map((c) => `${c.name}: $${c.price}`).join('\n') || 'No data'}

Provide pricing recommendation in JSON (NO markdown):
{
  "recommendedPrice": number,
  "minPrice": number,
  "maxPrice": number,
  "strategy": "penetration|value|premium",
  "reasoning": "explanation",
  "profitMargin": number,
  "competitivePosition": "undercut|match|premium"
}
`;

      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: prompt }],
        temperature: 0.6,
        max_tokens: 300
      });

      const jsonMatch = response.choices[0].message.content?.match(/\{[\s\S]*\}/);
      const pricing = jsonMatch ? JSON.parse(jsonMatch[0]) : {};

      // সংরক্ষণ করুন
      // await this.prisma.pricingAnalysis.create({...});

      return { success: true, data: pricing };
    } catch (err) {
      this.logger.error('Pricing analysis failed:', err);
      throw err;
    }
  }
}