import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class ProductGeneratorService {
  private logger = new Logger('ProductGeneratorService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // পণ্যের শিরোনাম, বর্ণনা, ট্যাগ তৈরি করুন
  async generateProductContent(tenantId: string, input: {
    productName: string;
    category: string;
    price: number;
    features?: string[];
  }) {
    try {
      const [title, description, tags, seoMeta] = await Promise.all([
        this.generateTitle(input.productName, input.category),
        this.generateDescription(input.productName, input.features || []),
        this.generateTags(input.productName, input.category),
        this.generateSeoMeta(input.productName, input.category)
      ]);

      // সেভ করুন
      const content = await this.prisma.aiGeneratedContent.create({
        data: {
          tenantId,
          type: 'product_full',
          content: JSON.stringify({ title, description, tags, seoMeta }),
          variations: [title, description],
          model: 'gpt-3.5-turbo',
          tokens: 500 // Approximate
        }
      });

      return {
        success: true,
        data: {
          title,
          description,
          tags,
          seoMeta,
          contentId: content.id
        }
      };
    } catch (err) {
      this.logger.error('Product generation failed:', err);
      throw err;
    }
  }

  private async generateTitle(productName: string, category: string): Promise<string> {
    const prompt = `
Generate a catchy, SEO-friendly product title for:
Product: ${productName}
Category: ${category}

Requirements:
- 50-70 characters max
- Include key benefits
- Engaging and clickable

Return ONLY the title, no quotes.
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: prompt }],
      temperature: 0.8,
      max_tokens: 50
    });

    return response.choices[0].message.content?.trim() || productName;
  }

  private async generateDescription(productName: string, features: string[]): Promise<string> {
    const featuresList = features.join(', ');
    const prompt = `
Write a compelling product description for "${productName}".
Key features: ${featuresList}

Requirements:
- 3-4 sentences
- Include benefits, not just features
- Call to action
- Engaging tone

Return ONLY the description.
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: prompt }],
      temperature: 0.7,
      max_tokens: 200
    });

    return response.choices[0].message.content?.trim() || '';
  }

  private async generateTags(productName: string, category: string): Promise<string[]> {
    const prompt = `
Generate 5-10 relevant tags for product "${productName}" in category "${category}".
Format: comma-separated, lowercase

Tags should be:
- Search friendly
- Relevant to product
- Popular on e-commerce sites

Return ONLY comma-separated tags.
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: prompt }],
      temperature: 0.6,
      max_tokens: 100
    });

    const tagsString = response.choices[0].message.content?.trim() || '';
    return tagsString.split(',').map((tag) => tag.trim());
  }

  private async generateSeoMeta(productName: string, category: string) {
    const prompt = `
Generate SEO metadata for product "${productName}" in "${category}".

Return JSON (NO markdown):
{
  "seoTitle": "SEO title (60 chars max)",
  "seoDescription": "Meta description (160 chars max)",
  "seoKeywords": "5-7 keywords comma-separated"
}
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: prompt }],
      temperature: 0.5,
      max_tokens: 200
    });

    try {
      const jsonMatch = response.choices[0].message.content?.match(/\{[\s\S]*\}/);
      return jsonMatch ? JSON.parse(jsonMatch[0]) : {};
    } catch (err) {
      return { seoTitle: productName, seoDescription: '', seoKeywords: '' };
    }
  }
}