import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class StoreWizardService {
  private logger = new Logger('StoreWizardService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // AI দিয়ে স্টোর সেটআপ উইজার্ড
  async setupStoreWithAi(tenantId: string, input: {
    businessName: string;
    description: string;
    targetAudience?: string;
  }) {
    try {
      this.logger.log(`AI Store Setup started for ${tenantId}`);

      // ১. Niche ডিটেক্ট করুন
      const niche = await this.detectNiche(input.description);

      // ২. কালার স্কিম জেনারেট করুন
      const colors = await this.generateColorScheme(niche);

      // ३. লোগো জেনারেট করুন (DALL-E)
      const logoUrl = await this.generateLogo(input.businessName, niche);

      // ४. থিম সিলেক্ট করুন
      const recommendedTheme = this.getRecommendedTheme(niche);

      // ५. সেভ করুন
      const theme = await this.prisma.storeTheme.upsert({
        where: { tenantId },
        create: {
          tenantId,
          logoUrl,
          primaryColor: colors.primary,
          secondaryColor: colors.secondary,
          niche,
          autoDetected: true
        },
        update: {
          logoUrl,
          primaryColor: colors.primary,
          secondaryColor: colors.secondary,
          niche,
          autoDetected: true
        }
      });

      return {
        success: true,
        data: {
          niche,
          colors,
          logoUrl,
          theme: recommendedTheme,
          setupCompleted: true
        }
      };
    } catch (err) {
      this.logger.error('Store setup failed:', err);
      throw err;
    }
  }

  // Niche ডিটেক্ট করুন
  private async detectNiche(description: string): Promise<string> {
    const prompt = `
Based on this business description, identify the niche/category:
"${description}"

Return ONLY one word niche from: electronics, fashion, food, home, beauty, health, sports, books, toys, etc.
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: prompt }],
      temperature: 0.5,
      max_tokens: 20
    });

    return response.choices[0].message.content?.trim().toLowerCase() || 'general';
  }

  // কালার স্কিম জেনারেট করুন
  private async generateColorScheme(niche: string): Promise<{ primary: string; secondary: string }> {
    const colorSchemes: any = {
      electronics: { primary: '#1a73e8', secondary: '#34a853' },
      fashion: { primary: '#e62e04', secondary: '#fbbc04' },
      food: { primary: '#d32f2f', secondary: '#ff9800' },
      beauty: { primary: '#c2185b', secondary: '#9c27b0' },
      health: { primary: '#00796b', secondary: '#4caf50' },
      sports: { primary: '#1976d2', secondary: '#0097a7' },
      home: { primary: '#6d4c41', secondary: '#8d6e63' },
      books: { primary: '#3e2723', secondary: '#795548' },
      toys: { primary: '#5e35b1', secondary: '#673ab7' },
      default: { primary: '#007bff', secondary: '#6c757d' }
    };

    return colorSchemes[niche] || colorSchemes.default;
  }

  // DALL-E দিয়ে লোগো জেনারেট করুন
  private async generateLogo(businessName: string, niche: string): Promise<string> {
    try {
      const prompt = `
Create a professional, minimalist logo for a ${niche} business called "${businessName}".
Style: Modern, clean, memorable. PNG format.
`;

      // Note: DALL-E image generation requires image URL storage
      // For now, we'll use a placeholder
      // In production, use DALL-E API or design service
      
      const logoUrl = `https://via.placeholder.com/200?text=${encodeURIComponent(businessName.slice(0, 2))}`;

      return logoUrl;
    } catch (err) {
      this.logger.warn('Logo generation failed, using default');
      return 'https://via.placeholder.com/200?text=Logo';
    }
  }

  // Recommended থিম পান
  private getRecommendedTheme(niche: string): string {
    const themeMap: any = {
      electronics: 'modern',
      fashion: 'elegant',
      food: 'warm',
      beauty: 'luxurious',
      health: 'clean',
      sports: 'dynamic',
      home: 'cozy',
      books: 'minimalist',
      toys: 'playful'
    };

    return themeMap[niche] || 'default';
  }
}