import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class AdvancedAnalyticsService {
  constructor(private prisma: PrismaService) {}

  // মাল্টি-ডিমেনশনাল অ্যানালিটিক্স
  async getAdvancedAnalytics(tenantId: string, filters: any) {
    const orders = await this.prisma.order.findMany({
      where: {
        tenantId,
        createdAt: {
          gte: filters.startDate,
          lte: filters.endDate
        }
      },
      include: { items: true, user: true }
    });

    // বিভিন্ন মেট্রিক্স ক্যালকুলেট করুন
    const metrics = {
      totalRevenue: orders.reduce((sum, o) => sum + o.total, 0),
      totalOrders: orders.length,
      avgOrderValue: orders.length > 0 ? orders.reduce((sum, o) => sum + o.total, 0) / orders.length : 0,
      conversionRate: 0, // TODO: Calculate from page views
      
      // পণ্য পারফরম্যান্স
      topProducts: this.getTopProducts(orders),
      topCategories: this.getTopCategories(orders),
      
      // গ্রাহক মেট্রিক্স
      newCustomers: await this.getNewCustomersCount(tenantId, filters.startDate, filters.endDate),
      repeatCustomers: await this.getRepeatCustomersCount(tenantId, filters.startDate, filters.endDate),
      
      // সময় ভিত্তিক
      dailyRevenue: this.getDailyRevenue(orders),
      hourlyOrders: this.getHourlyOrders(orders)
    };

    return { success: true, data: metrics };
  }

  private getTopProducts(orders: any[]) {
    const products: any = {};

    for (const order of orders) {
      for (const item of order.items) {
        if (!products[item.productId]) {
          products[item.productId] = { quantity: 0, revenue: 0 };
        }
        products[item.productId].quantity += item.quantity;
        products[item.productId].revenue += item.total;
      }
    }

    return Object.entries(products)
      .sort((a: any, b: any) => b[1].revenue - a[1].revenue)
      .slice(0, 10);
  }

  private getTopCategories(orders: any[]) {
    // ক্যাটেগরি পারফরম্যান্স
    return [];
  }

  private getDailyRevenue(orders: any[]) {
    const daily: any = {};

    for (const order of orders) {
      const date = new Date(order.createdAt).toISOString().split('T')[0];
      if (!daily[date]) daily[date] = 0;
      daily[date] += order.total;
    }

    return Object.entries(daily).map(([date, revenue]) => ({ date, revenue }));
  }

  private getHourlyOrders(orders: any[]) {
    const hourly: any = {};

    for (const order of orders) {
      const hour = new Date(order.createdAt).getHours();
      if (!hourly[hour]) hourly[hour] = 0;
      hourly[hour]++;
    }

    return Object.entries(hourly).map(([hour, count]) => ({ hour, count }));
  }

  private async getNewCustomersCount(
    tenantId: string,
    startDate: Date,
    endDate: Date
  ): Promise<number> {
    return await this.prisma.user.count({
      where: {
        tenantId,
        createdAt: { gte: startDate, lte: endDate }
      }
    });
  }

  private async getRepeatCustomersCount(
    tenantId: string,
    startDate: Date,
    endDate: Date
  ): Promise<number> {
    const users = await this.prisma.user.findMany({
      where: { tenantId },
      include: {
        orders: {
          where: {
            createdAt: { gte: startDate, lte: endDate }
          }
        }
      }
    });

    return users.filter((u) => u.orders.length > 1).length;
  }
}