import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class AIDashboardService {
  private logger = new Logger('AIDashboardService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // AI-পাওয়ারড ড্যাশবোর্ড সারাংশ জেনারেট করুন
  async generateAIDashboardInsights(tenantId: string) {
    try {
      // সব মেট্রিক্স সংগ্রহ করুন
      const metrics = await this.collectDashboardMetrics(tenantId);

      // AI দিয়ে ইনসাইট তৈরি করুন
      const insights = await this.generateInsightsWithAI(metrics);

      return {
        success: true,
        data: {
          metrics,
          insights,
          recommendations: insights.recommendations
        }
      };
    } catch (err) {
      this.logger.error('Dashboard generation failed:', err);
      throw err;
    }
  }

  private async collectDashboardMetrics(tenantId: string) {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const lastMonth = new Date(today);
    lastMonth.setDate(lastMonth.getDate() - 30);

    // পেজভিউ
    const pageViews = await this.prisma.analyticsEvent.count({
      where: {
        tenantId,
        eventType: 'page_view',
        createdAt: { gte: today }
      }
    });

    // ইউনিক ভিজিটর
    const uniqueVisitors = await this.prisma.analyticsEvent.findMany({
      where: {
        tenantId,
        createdAt: { gte: today }
      },
      distinct: ['sessionId'],
      select: { sessionId: true }
    });

    // কনভার্সন
    const orders = await this.prisma.order.count({
      where: {
        tenantId,
        createdAt: { gte: today }
      }
    });

    // রেভিনিউ
    const revenue = await this.prisma.order.aggregate({
      where: {
        tenantId,
        createdAt: { gte: today }
      },
      _sum: { total: true }
    });

    // বাউন্স রেট (গণনা করুন)
    const sessionDurations = await this.prisma.analyticsEvent.findMany({
      where: {
        tenantId,
        createdAt: { gte: today }
      },
      select: { sessionId: true, timeOnPage: true }
    });

    const bounceRate = this.calculateBounceRate(sessionDurations);

    // গড় অর্ডার ভ্যালু
    const avgOrderValue =
      orders > 0 ? (Number(revenue._sum.total) || 0) / orders : 0;

    return {
      pageViews,
      uniqueVisitors: uniqueVisitors.length,
      orders,
      revenue: Number(revenue._sum.total) || 0,
      bounceRate,
      avgOrderValue,
      conversionRate: uniqueVisitors.length > 0
        ? (orders / uniqueVisitors.length) * 100
        : 0
    };
  }

  private async generateInsightsWithAI(metrics: any) {
    const prompt = `
Analyze these e-commerce metrics and provide insights:
- Page Views: ${metrics.pageViews}
- Unique Visitors: ${metrics.uniqueVisitors}
- Orders: ${metrics.orders}
- Revenue: $${metrics.revenue.toFixed(2)}
- Bounce Rate: ${metrics.bounceRate.toFixed(2)}%
- Average Order Value: $${metrics.avgOrderValue.toFixed(2)}
- Conversion Rate: ${metrics.conversionRate.toFixed(2)}%

Provide:
1. Key performance analysis
2. Areas of concern
3. Actionable recommendations
4. Growth opportunities

Format as JSON with keys: summary, concerns, recommendations, opportunities
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: prompt }],
      temperature: 0.7,
      max_tokens: 1000
    });

    const content = response.choices[0].message.content || '{}';

    try {
      return JSON.parse(content);
    } catch {
      return {
        summary: content,
        recommendations: ['Review analytics data']
      };
    }
  }

  private calculateBounceRate(sessionDurations: any[]): number {
    const bouncedSessions = sessionDurations.filter(
      (s) => !s.timeOnPage || s.timeOnPage < 5
    ).length;

    return sessionDurations.length > 0
      ? (bouncedSessions / sessionDurations.length) * 100
      : 0;
  }
}