import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CustomerBehaviorService {
  constructor(private prisma: PrismaService) {}

  // কাস্টমার বিহেভিয়ার ট্র্যাক করুন
  async trackCustomerJourney(userId: string, action: string, data: any) {
    await this.prisma.customerBehavior.create({
      data: {
        userId,
        action, // "view_product", "add_cart", "checkout", "purchase"
        productId: data.productId,
        categoryId: data.categoryId,
        timeSpent: data.timeSpent,
        metadata: data
      }
    });
  }

  // চার্ন রিস্ক প্রেডিক্ট করুন (AI)
  async predictChurnRisk(userId: string): Promise<{ risk: number; reason: string }> {
    const customer = await this.prisma.customer.findUnique({
      where: { userId },
      include: { orders: true }
    });

    if (!customer || customer.orders.length === 0) {
      return { risk: 0, reason: 'New customer' };
    }

    const daysSinceLastOrder = Math.floor(
      (Date.now() - new Date(customer.orders[customer.orders.length - 1].createdAt).getTime()) /
        (1000 * 60 * 60 * 24)
    );

    let riskScore = 0;
    let reason = '';

    if (daysSinceLastOrder > 90) {
      riskScore = 0.9;
      reason = 'No purchase in 90+ days';
    } else if (daysSinceLastOrder > 60) {
      riskScore = 0.7;
      reason = 'No purchase in 60+ days';
    } else if (daysSinceLastOrder > 30) {
      riskScore = 0.5;
      reason = 'No purchase in 30+ days';
    }

    return { risk: riskScore, reason };
  }

  // পার্সোনালাইজড রিকমেন্ডেশন
  async getPersonalizedRecommendations(userId: string) {
    const customer = await this.prisma.customer.findUnique({
      where: { userId },
      include: {
        orders: { include: { items: { include: { product: true } } } }
      }
    });

    if (!customer) return [];

    // সবচেয়ে দেখা ক্যাটাগরি খুঁজুন
    const categoryMap: any = {};
    customer.orders.forEach((order) => {
      order.items.forEach((item) => {
        const cat = item.product.categoryId;
        categoryMap[cat] = (categoryMap[cat] || 0) + 1;
      });
    });

    const topCategory = Object.keys(categoryMap).sort(
      (a, b) => categoryMap[b] - categoryMap[a]
    )[0];

    // সেই ক্যাটাগরির পণ্য সুপারিশ করুন
    const recommendations = await this.prisma.product.findMany({
      where: { categoryId: topCategory },
      take: 5
    });

    return recommendations;
  }
}