import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as ExcelJS from 'exceljs';
import * as PDFDocument from 'pdfkit';
import * as AWS from 'aws-sdk';

@Injectable()
export class ReportExportService {
  private logger = new Logger('ReportExportService');
  private s3 = new AWS.S3();

  constructor(private prisma: PrismaService) {}

  // রিপোর্ট PDF-তে এক্সপোর্ট করুন
  async exportReportToPDF(tenantId: string, reportData: any) {
    try {
      const doc = new PDFDocument();
      const fileName = `/tmp/report-${Date.now()}.pdf`;

      // হেডার
      doc.fontSize(20).text('Analytics Report', 100, 100);
      doc.fontSize(10).text(`Generated: ${new Date().toLocaleDateString()}`);

      // কনটেন্ট
      doc.fontSize(12).text('Key Metrics:', 100, 200);
      Object.keys(reportData).forEach((key, index) => {
        doc.text(`${key}: ${reportData[key]}`, 100, 230 + index * 20);
      });

      doc.pipe(require('fs').createWriteStream(fileName));
      doc.end();

      // S3-এ আপলোড করুন
      const s3Url = await this.uploadToS3(fileName, 'pdf');

      return { success: true, url: s3Url };
    } catch (err) {
      this.logger.error('PDF export failed:', err);
      throw err;
    }
  }

  // রিপোর্ট Excel-এ এক্সপোর্ট করুন
  async exportReportToExcel(tenantId: string, reportData: any) {
    try {
      const workbook = new ExcelJS.Workbook();
      const worksheet = workbook.addWorksheet('Report');

      // হেডার
      worksheet.columns = Object.keys(reportData[0] || {}).map((key) => ({
        header: key,
        key: key
      }));

      // ডেটা
      reportData.forEach((row: any) => {
        worksheet.addRow(row);
      });

      const fileName = `/tmp/report-${Date.now()}.xlsx`;
      await workbook.xlsx.writeFile(fileName);

      // S3-এ আপলোড করুন
      const s3Url = await this.uploadToS3(fileName, 'xlsx');

      return { success: true, url: s3Url };
    } catch (err) {
      this.logger.error('Excel export failed:', err);
      throw err;
    }
  }

  private async uploadToS3(
    filePath: string,
    format: 'pdf' | 'xlsx'
  ): Promise<string> {
    const fileContent = require('fs').readFileSync(filePath);
    const key = `reports/${Date.now()}.${format}`;

    await this.s3
      .putObject({
        Bucket: process.env.AWS_S3_BUCKET || '',
        Key: key,
        Body: fileContent,
        ContentType:
          format === 'pdf'
            ? 'application/pdf'
            : 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
      })
      .promise();

    return `https://${process.env.AWS_S3_BUCKET}.s3.amazonaws.com/${key}`;
  }

  // জেনারেট করা রিপোর্ট সংরক্ষণ করুন
  async saveGeneratedReport(
    tenantId: string,
    reportType: string,
    reportData: any,
    format: 'pdf' | 'excel'
  ) {
    const fileUrl =
      format === 'pdf'
        ? await this.exportReportToPDF(tenantId, reportData)
        : await this.exportReportToExcel(tenantId, reportData);

    const expiresAt = new Date();
    expiresAt.setDate(expiresAt.getDate() + 30); // ৩০ দিন

    const report = await this.prisma.generatedReport.create({
      data: {
        tenantId,
        reportType,
        reportData,
        format,
        fileUrl: fileUrl.url,
        expiresAt
      }
    });

    return { success: true, reportId: report.id, fileUrl: fileUrl.url };
  }
}