import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class TopProductsAnalyticsService {
  constructor(private prisma: PrismaService) {}

  // শীর্ষ পণ্য বিশ্লেষণ পান
  async getTopProducts(tenantId: string, period: 'daily' | 'weekly' | 'monthly' = 'daily', limit = 10) {
    const periodDate = new Date();

    if (period === 'weekly') {
      periodDate.setDate(periodDate.getDate() - 7);
    } else if (period === 'monthly') {
      periodDate.setDate(periodDate.getDate() - 30);
    }

    const topProducts = await this.prisma.topProductAnalytics.findMany({
      where: {
        tenantId,
        period,
        periodDate: { gte: periodDate }
      },
      include: { product: true },
      orderBy: { revenue: 'desc' },
      take: limit
    });

    return {
      period,
      products: topProducts.map((p) => ({
        productId: p.productId,
        productName: p.product.title,
        views: p.views,
        clicks: p.clicks,
        purchases: p.purchases,
        revenue: p.revenue,
        conversionRate: p.conversionRate
      }))
    };
  }

  // পণ্য পারফরম্যান্স ট্র্যাক করুন
  async trackProductPerformance(tenantId: string) {
    const products = await this.prisma.product.findMany({
      where: { tenantId }
    });

    for (const product of products) {
      // ভিউ কাউন্ট
      const views = await this.prisma.analyticsEvent.count({
        where: {
          tenantId,
          pageUrl: { contains: product.id }
        }
      });

      // ক্রয় কাউন্ট
      const purchases = await this.prisma.order.count({
        where: {
          tenantId,
          items: { some: { productId: product.id } }
        }
      });

      // রেভিনিউ
      const orderItems = await this.prisma.orderItem.findMany({
        where: { productId: product.id }
      });

      const revenue = orderItems.reduce(
        (sum, item) => sum + item.price * item.quantity,
        0
      );

      const conversionRate = views > 0 ? (purchases / views) * 100 : 0;

      // রেকর্ড করুন
      await this.prisma.topProductAnalytics.upsert({
        where: {
          tenantId_productId_period_periodDate: {
            tenantId,
            productId: product.id,
            period: 'daily',
            periodDate: new Date()
          }
        } as any,
        create: {
          tenantId,
          productId: product.id,
          views,
          clicks: views, // পারফরম্যান্সের জন্য সমান
          purchases,
          revenue: BigInt(Math.round(revenue)) as any,
          conversionRate,
          period: 'daily',
          periodDate: new Date()
        },
        update: {
          views,
          clicks: views,
          purchases,
          revenue: BigInt(Math.round(revenue)) as any,
          conversionRate
        }
      });
    }
  }
}