import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class ProductApprovalService {
  private logger = new Logger('ProductApprovalService');

  constructor(private prisma: PrismaService) {}

  // প্রোডাক্ট অনুমোদনের জন্য সাবমিট করুন
  async submitForApproval(wholesalerTenantId: string, productId: string) {
    const approval = await this.prisma.productApproval.create({
      data: {
        productId,
        tenantId: wholesalerTenantId,
        status: 'pending'
      }
    });

    this.logger.log(`Product submitted for approval: ${productId}`);

    return { success: true, data: approval };
  }

  // প্রোডাক্ট অনুমোদন করুন
  async approveProduct(approvalId: string, adminId: string) {
    const approval = await this.prisma.productApproval.update({
      where: { id: approvalId },
      data: {
        status: 'approved',
        reviewedBy: adminId,
        reviewedAt: new Date()
      }
    });

    // সাবস্ক্রাইবড স্টোরে প্রোডাক্ট যোগ করুন (যদি স্বয়ংক্রিয় অনুমোদন অন থাকে)
    await this.distributeToSubscribedStores(approval.tenantId, approval.productId);

    return { success: true, data: approval };
  }

  // প্রোডাক্ট প্রত্যাখ্যান করুন
  async rejectProduct(approvalId: string, adminId: string, reason: string) {
    const approval = await this.prisma.productApproval.update({
      where: { id: approvalId },
      data: {
        status: 'rejected',
        rejectionReason: reason,
        reviewedBy: adminId,
        reviewedAt: new Date()
      }
    });

    // প্রত্যাখ্যানের কারণ পাঠান
    await this.notifyRejection(approval.tenantId, approval.productId, reason);

    return { success: true, data: approval };
  }

  private async distributeToSubscribedStores(wholesalerTenantId: string, productId: string) {
    const connections = await this.prisma.adminWholesalerConnection.findMany({
      where: { wholesalerTenantId, status: 'active' }
    });

    const product = await this.prisma.product.findUnique({
      where: { id: productId }
    });

    if (!product) return;

    for (const connection of connections) {
      // প্রোডাক্ট স্টোরে যোগ করুন
      await this.prisma.product.create({
        data: {
          tenantId: connection.adminTenantId,
          sku: product.sku,
          title: product.title,
          description: product.description,
          price: product.price,
          stock: product.stock,
          images: product.images
        }
      });
    }
  }

  private async notifyRejection(wholesalerTenantId: string, productId: string, reason: string) {
    // ইমেইল পাঠান
    this.logger.log(`Rejection notification: ${reason}`);
  }

  // পেন্ডিং অনুমোদন পান
  async getPendingApprovals(limit = 50) {
    return await this.prisma.productApproval.findMany({
      where: { status: 'pending' },
      take: limit,
      orderBy: { submittedAt: 'asc' }
    });
  }
}