import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class AuditService {
  private logger = new Logger('AuditService');

  constructor(private prisma: PrismaService) {}

  // কী: সব অ্যাকশন লগ করুন
  async logAction(
    tenantId: string,
    userId: string,
    action: string,
    entityType: string,
    entityId: string,
    oldValues?: any,
    newValues?: any,
    ipAddress?: string,
    userAgent?: string
  ) {
    const log = await this.prisma.auditLog.create({
      data: {
        tenantId,
        action,
        entityType,
        entityId,
        userId,
        oldValues,
        newValues,
        ipAddress,
        userAgent
      }
    });

    this.logger.log(
      `Audit: ${action} - ${entityType}#${entityId} by user#${userId}`
    );

    return log;
  }

  // অডিট লগ পান
  async getLogs(tenantId: string, skip = 0, take = 100, filters?: any) {
    const where: any = { tenantId };

    if (filters?.entityType) where.entityType = filters.entityType;
    if (filters?.action) where.action = filters.action;
    if (filters?.userId) where.userId = filters.userId;
    if (filters?.startDate) {
      where.createdAt = { gte: new Date(filters.startDate) };
    }

    const [logs, total] = await Promise.all([
      this.prisma.auditLog.findMany({
        where,
        skip,
        take,
        orderBy: { createdAt: 'desc' }
      }),
      this.prisma.auditLog.count({ where })
    ]);

    return {
      items: logs,
      total,
      page: Math.floor(skip / take) + 1,
      limit: take
    };
  }

  // কোন পরিবর্তন হয়েছে তা ট্র্যাক করুন
  async trackChange(
    tenantId: string,
    userId: string,
    entityType: string,
    entityId: string,
    oldValues: any,
    newValues: any,
    req?: any
  ) {
    const changes: any = {};

    for (const key in newValues) {
      if (oldValues[key] !== newValues[key]) {
        changes[key] = {
          from: oldValues[key],
          to: newValues[key]
        };
      }
    }

    if (Object.keys(changes).length === 0) {
      return; // No changes
    }

    await this.logAction(
      tenantId,
      userId,
      `${entityType.toLowerCase()}.updated`,
      entityType,
      entityId,
      oldValues,
      newValues,
      req?.ip,
      req?.headers['user-agent']
    );
  }
}