import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as speakeasy from 'speakeasy';
import * as QRCode from 'qrcode';

@Injectable()
export class TwoFactorAuthService {
  private logger = new Logger('TwoFactorAuthService');

  constructor(private prisma: PrismaService) {}

  // 2FA সক্ষম করুন
  async enable2FA(userId: string) {
    // সিক্রেট জেনারেট করুন
    const secret = speakeasy.generateSecret({
      name: `YourApp (${userId})`,
      issuer: 'YourApp',
      length: 32
    });

    // QR কোড জেনারেট করুন
    const qrCode = await QRCode.toDataURL(secret.otpauth_url);

    // ব্যাকআপ কোড জেনারেট করুন
    const backupCodes = Array.from({ length: 10 }, () =>
      Math.random().toString(36).substring(2, 10).toUpperCase()
    );

    // এখনো সংরক্ষণ করবেন না - ভেরিফিকেশনের জন্য অপেক্ষা করুন

    return {
      secret: secret.base32,
      qrCode,
      backupCodes
    };
  }

  // 2FA ভেরিফাই করুন
  async verify2FA(
    userId: string,
    token: string,
    secret: string
  ): Promise<boolean> {
    const isValid = speakeasy.totp.verify({
      secret,
      encoding: 'base32',
      token,
      window: 2
    });

    if (isValid) {
      // ডাটাবেসে সংরক্ষণ করুন
      const backupCodes = Array.from({ length: 10 }, () =>
        Math.random().toString(36).substring(2, 10).toUpperCase()
      );

      await this.prisma.twoFactorAuth.upsert({
        where: { userId },
        create: {
          userId,
          secret,
          isEnabled: true,
          backupCodes
        },
        update: {
          secret,
          isEnabled: true,
          backupCodes
        }
      });

      this.logger.log(`2FA enabled for user ${userId}`);
    }

    return isValid;
  }

  // 2FA চেক করুন (লগইনের সময়)
  async verify2FAToken(userId: string, token: string): Promise<boolean> {
    const twoFactorAuth = await this.prisma.twoFactorAuth.findUnique({
      where: { userId }
    });

    if (!twoFactorAuth || !twoFactorAuth.isEnabled) {
      return true; // 2FA সক্ষম নয়
    }

    // TOTP যাচাই করুন
    const isValid = speakeasy.totp.verify({
      secret: twoFactorAuth.secret,
      encoding: 'base32',
      token,
      window: 2
    });

    if (isValid) {
      return true;
    }

    // ব্যাকআপ কোড যাচাই করুন
    if (twoFactorAuth.backupCodes.includes(token)) {
      // ব্যবহৃত কোড সরান
      const updatedCodes = twoFactorAuth.backupCodes.filter(
        (code) => code !== token
      );

      await this.prisma.twoFactorAuth.update({
        where: { userId },
        data: { backupCodes: updatedCodes }
      });

      return true;
    }

    return false;
  }

  // 2FA অক্ষম করুন
  async disable2FA(userId: string) {
    await this.prisma.twoFactorAuth.update({
      where: { userId },
      data: { isEnabled: false }
    });

    this.logger.log(`2FA disabled for user ${userId}`);

    return { success: true };
  }
}