import { Controller, Post, Body, Get, UseGuards, Req, BadRequestException } from '@nestjs/common';
import { ApiTags, ApiBearerAuth } from '@nestjs/swagger';
import { AuthService } from './auth.service';
import { RegisterDto } from './dto/register.dto';
import { LoginDto } from './dto/login.dto';
import { JwtAuthGuard } from './guards/jwt-auth.guard';

@ApiTags('Auth')
@Controller('auth')
export class AuthController {
  constructor(private authService: AuthService) {}

  @Post('register')
  async register(@Body() dto: RegisterDto) {
    return await this.authService.register(dto);
  }

  @Post('login')
  async login(@Body() dto: LoginDto) {
    return await this.authService.login(dto);
  }

  @Post('verify-email')
  async verifyEmail(@Body('token') token: string) {
    if (!token) throw new BadRequestException('Token required');
    return await this.authService.verifyEmail(token);
  }

  @Post('2fa/setup')
  @UseGuards(JwtAuthGuard)
  async setupTwoFactor(@Req() req: any) {
    return await this.authService.setupTwoFactor(req.user.sub);
  }

  @Post('2fa/confirm')
  @UseGuards(JwtAuthGuard)
  async confirmTwoFactor(@Req() req: any, @Body() body: { secret: string; token: string }) {
    return await this.authService.confirmTwoFactor(req.user.sub, body.secret, body.token);
  }

  @Post('2fa/verify')
  async verifyTwoFactor(@Body() body: { userId: string; token: string }) {
    return await this.authService.verifyTwoFactor(body.userId, body.token);
  }

  @Post('password-reset/request')
  async requestPasswordReset(@Body() body: { email: string; tenantId: string }) {
    return await this.authService.requestPasswordReset(body.email, body.tenantId);
  }

  @Post('password-reset/confirm')
  async resetPassword(@Body() body: { token: string; newPassword: string }) {
    return await this.authService.resetPassword(body.token, body.newPassword);
  }

  @Get('me')
  @UseGuards(JwtAuthGuard)
  async getMe(@Req() req: any) {
    return { data: req.user };
  }
}