import { Injectable, BadRequestException, UnauthorizedException, ConflictException } from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import { PrismaService } from '../../database/prisma.service';
import { RegisterDto } from './dto/register.dto';
import { LoginDto } from './dto/login.dto';
import * as bcrypt from 'bcrypt';
import * as speakeasy from 'speakeasy';
import * as QRCode from 'qrcode';

@Injectable()
export class AuthService {
  constructor(
    private prisma: PrismaService,
    private jwtService: JwtService
  ) {}

  async register(dto: RegisterDto) {
    const { email, password, name, tenantId } = dto;

    // Check if user exists
    const existing = await this.prisma.user.findFirst({
      where: { email, tenantId }
    });

    if (existing) {
      throw new ConflictException('Email already registered');
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 10);

    const user = await this.prisma.user.create({
      data: {
        email,
        name,
        password: hashedPassword,
        tenantId,
        role: 'CUSTOMER',
        status: 'active'
      }
    });

    // Send verification email (queue this)
    await this.sendVerificationEmail(user.email, user.id);

    return {
      success: true,
      message: 'Registration successful. Check your email to verify.',
      data: { id: user.id, email: user.email, name: user.name }
    };
  }

  async login(dto: LoginDto) {
    const { email, password, tenantId } = dto;

    const user = await this.prisma.user.findFirst({
      where: { email, tenantId }
    });

    if (!user) {
      throw new UnauthorizedException('Invalid email or password');
    }

    const isValid = await bcrypt.compare(password, user.password);
    if (!isValid) {
      throw new UnauthorizedException('Invalid email or password');
    }

    if (!user.emailVerified) {
      throw new UnauthorizedException('Please verify your email first');
    }

    // Check if 2FA is enabled
    if (user.twoFactorEnabled) {
      return {
        success: true,
        requiresTwoFactor: true,
        tempToken: this.jwtService.sign(
          { sub: user.id, tenantId, tempAuth: true },
          { expiresIn: '5m' }
        )
      };
    }

    return this.generateTokens(user);
  }

  async verifyEmail(token: string) {
    try {
      const decoded = this.jwtService.verify(token, { secret: process.env.JWT_EMAIL_SECRET });

      await this.prisma.user.update({
        where: { id: decoded.sub },
        data: { emailVerified: true }
      });

      return { success: true, message: 'Email verified' };
    } catch (err) {
      throw new BadRequestException('Invalid or expired token');
    }
  }

  // 2FA Setup
  async setupTwoFactor(userId: string) {
    const secret = speakeasy.generateSecret({
      name: `SaaS (${userId})`
    });

    const qrCode = await QRCode.toDataURL(secret.otpauth_url);

    return {
      secret: secret.base32,
      qrCode
    };
  }

  async confirmTwoFactor(userId: string, secret: string, token: string) {
    const isValid = speakeasy.totp.verify({
      secret,
      encoding: 'base32',
      token
    });

    if (!isValid) {
      throw new BadRequestException('Invalid 2FA token');
    }

    await this.prisma.user.update({
      where: { id: userId },
      data: {
        twoFactorEnabled: true,
        twoFactorSecret: secret
      }
    });

    return { success: true, message: '2FA enabled' };
  }

  async verifyTwoFactor(userId: string, token: string) {
    const user = await this.prisma.user.findUnique({ where: { id: userId } });

    const isValid = speakeasy.totp.verify({
      secret: user.twoFactorSecret,
      encoding: 'base32',
      token
    });

    if (!isValid) {
      throw new UnauthorizedException('Invalid 2FA token');
    }

    return this.generateTokens(user);
  }

  async requestPasswordReset(email: string, tenantId: string) {
    const user = await this.prisma.user.findFirst({ where: { email, tenantId } });

    if (!user) {
      // Don't reveal if email exists
      return { success: true, message: 'If email exists, reset link sent' };
    }

    const resetToken = this.jwtService.sign(
      { sub: user.id, purpose: 'reset' },
      { expiresIn: '30m', secret: process.env.JWT_EMAIL_SECRET }
    );

    // Queue email
    await this.sendPasswordResetEmail(user.email, resetToken);

    return { success: true, message: 'Reset link sent' };
  }

  async resetPassword(token: string, newPassword: string) {
    try {
      const decoded = this.jwtService.verify(token, {
        secret: process.env.JWT_EMAIL_SECRET
      });

      if (decoded.purpose !== 'reset') {
        throw new BadRequestException('Invalid token');
      }

      const hashedPassword = await bcrypt.hash(newPassword, 10);

      await this.prisma.user.update({
        where: { id: decoded.sub },
        data: { password: hashedPassword }
      });

      return { success: true, message: 'Password reset successful' };
    } catch (err) {
      throw new BadRequestException('Invalid or expired token');
    }
  }

  private async generateTokens(user: any) {
    const accessToken = this.jwtService.sign(
      { sub: user.id, email: user.email, tenantId: user.tenantId, role: user.role },
      { expiresIn: '7d' }
    );

    const refreshToken = this.jwtService.sign(
      { sub: user.id, type: 'refresh' },
      { expiresIn: '30d', secret: process.env.JWT_REFRESH_SECRET }
    );

    return {
      success: true,
      data: {
        accessToken,
        refreshToken,
        user: {
          id: user.id,
          email: user.email,
          name: user.name,
          role: user.role
        }
      }
    };
  }

  private async sendVerificationEmail(email: string, userId: string) {
    const token = this.jwtService.sign(
      { sub: userId },
      { expiresIn: '24h', secret: process.env.JWT_EMAIL_SECRET }
    );

    // Queue to email worker
    console.log(`[EMAIL QUEUE] Verification email to ${email} with token ${token}`);
    // TODO: Push to BullMQ queue
  }

  private async sendPasswordResetEmail(email: string, token: string) {
    console.log(`[EMAIL QUEUE] Reset email to ${email} with token ${token}`);
    // TODO: Push to BullMQ queue
  }
}