import { Injectable, Logger } from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import { PrismaService } from '../../database/prisma.service';
import * as bcrypt from 'bcrypt';

@Injectable()
export class JWTAuthService {
  private logger = new Logger('JWTAuthService');

  constructor(
    private jwtService: JwtService,
    private prisma: PrismaService
  ) {}

  // লগইন করুন এবং JWT টোকেন তৈরি করুন
  async login(email: string, password: string) {
    const user = await this.prisma.user.findUnique({
      where: { email }
    });

    if (!user) {
      throw new Error('User not found');
    }

    // পাসওয়ার্ড যাচাই করুন
    const isPasswordValid = await bcrypt.compare(password, user.password);

    if (!isPasswordValid) {
      throw new Error('Invalid password');
    }

    // JWT টোকেন তৈরি করুন
    const payload = {
      userId: user.id,
      email: user.email,
      role: user.role
    };

    const accessToken = this.jwtService.sign(payload, {
      expiresIn: '24h'
    });

    const refreshToken = this.jwtService.sign(payload, {
      expiresIn: '7d'
    });

    // সেশন সংরক্ষণ করুন
    await this.prisma.session.create({
      data: {
        userId: user.id,
        token: accessToken,
        ipAddress: '0.0.0.0', // Request থেকে পান
        expiresAt: new Date(Date.now() + 24 * 60 * 60 * 1000)
      }
    });

    this.logger.log(`User logged in: ${email}`);

    return {
      accessToken,
      refreshToken,
      user: {
        id: user.id,
        email: user.email,
        name: user.name
      }
    };
  }

  // টোকেন রিফ্রেশ করুন
  async refreshToken(refreshToken: string) {
    try {
      const payload = this.jwtService.verify(refreshToken);

      const newAccessToken = this.jwtService.sign(
        {
          userId: payload.userId,
          email: payload.email,
          role: payload.role
        },
        { expiresIn: '24h' }
      );

      return { accessToken: newAccessToken };
    } catch (err) {
      throw new Error('Invalid refresh token');
    }
  }

  // লগআউট করুন (টোকেন ব্ল্যাকলিস্ট করুন)
  async logout(token: string) {
    const payload = this.jwtService.decode(token);

    if (payload && payload.exp) {
      await this.prisma.tokenBlacklist.create({
        data: {
          token,
          userId: payload.userId,
          expiresAt: new Date(payload.exp * 1000)
        }
      });
    }

    this.logger.log(`User logged out`);

    return { success: true };
  }

  // টোকেন ব্ল্যাকলিস্টে আছে কিনা চেক করুন
  async isTokenBlacklisted(token: string): Promise<boolean> {
    const blacklistedToken = await this.prisma.tokenBlacklist.findUnique({
      where: { token }
    });

    return !!blacklistedToken;
  }
}