import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';

@Injectable()
export class AffiliatePayoutAutomationService {
  private logger = new Logger('AffiliatePayoutAutomationService');

  constructor(private prisma: PrismaService) {}

  // সাপ্তাহিক অ্যাফিলিয়েট পেমেন্ট (প্রতি সোমবার)
  @Cron('0 10 * * 1')
  async autoProcessAffiliatePayouts() {
    this.logger.log('Starting automatic affiliate payouts...');

    const affiliates = await this.prisma.affiliateAccount.findMany({
      where: { status: 'active' }
    });

    for (const affiliate of affiliates) {
      try {
        // প্রকৃত কমিশন পান
        const commissions = await this.prisma.affiliateCommission.findMany({
          where: {
            affiliateId: affiliate.id,
            status: 'pending'
          }
        });

        const totalAmount = commissions.reduce(
          (sum, c) => sum + Number(c.amount),
          0
        );

        if (totalAmount > 0) {
          // পেমেন্ট ট্রিগার করুন
          await this.triggerPayment(affiliate, totalAmount, commissions);
        }
      } catch (err) {
        this.logger.error(`Payout failed for affiliate ${affiliate.id}:`, err);
      }
    }
  }

  private async triggerPayment(
    affiliate: any,
    amount: number,
    commissions: any[]
  ) {
    // পেমেন্ট মেথড অনুযায়ী পেমেন্ট করুন
    if (affiliate.paymentMethod === 'auto') {
      // স্বয়ংক্রিয় পেমেন্ট
      await this.processBkashPayment(affiliate, amount);
    }

    // সব কমিশন "paid" হিসেবে মার্ক করুন
    await this.prisma.affiliateCommission.updateMany({
      where: { id: { in: commissions.map((c) => c.id) } },
      data: { status: 'paid', paidAt: new Date() }
    });

    this.logger.log(`Affiliate payout processed: ${affiliate.id} - $${amount}`);
  }

  private async processBkashPayment(affiliate: any, amount: number) {
    // bKash পেমেন্ট প্রসেস
    // TODO: Implement bKash payment
    this.logger.log(`bKash payment: ${amount} to ${affiliate.id}`);
  }

  // ম্যানুয়াল পেমেন্ট ট্রিগার
  async triggerManualPayout(affiliateId: string) {
    const affiliate = await this.prisma.affiliateAccount.findUnique({
      where: { id: affiliateId }
    });

    if (!affiliate) throw new Error('Affiliate not found');

    const commissions = await this.prisma.affiliateCommission.findMany({
      where: { affiliateId, status: 'pending' }
    });

    const totalAmount = commissions.reduce(
      (sum, c) => sum + Number(c.amount),
      0
    );

    await this.triggerPayment(affiliate, totalAmount, commissions);

    return { success: true, amount: totalAmount };
  }
}