import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import axios from 'axios';

@Injectable()
export class APIImportAutomationService {
  private logger = new Logger('APIImportAutomationService');

  constructor(private prisma: PrismaService) {}

  // প্রতি ৬ ঘণ্টায় API থেকে পণ্য আমদানি করুন
  @Cron('0 */6 * * *')
  async autoImportFromAPIs() {
    this.logger.log('Starting API product imports...');

    const schedules = await this.prisma.aPIImportSchedule.findMany({
      where: { isActive: true }
    });

    for (const schedule of schedules) {
      try {
        await this.importProductsFromAPI(schedule);
      } catch (err) {
        this.logger.error(
          `API import failed for schedule ${schedule.id}:`,
          err
        );
      }
    }
  }

  private async importProductsFromAPI(schedule: any) {
    try {
      const response = await axios.get(schedule.apiUrl, {
        headers: schedule.apiKey ? { Authorization: `Bearer ${schedule.apiKey}` } : {}
      });

      const products = response.data.products || response.data;

      for (const productData of products) {
        await this.prisma.product.upsert({
          where: {
            tenantId_sku: {
              tenantId: schedule.tenantId,
              sku: productData.sku || productData.id
            }
          },
          create: {
            tenantId: schedule.tenantId,
            sku: productData.sku || productData.id,
            title: productData.title || productData.name,
            description: productData.description || '',
            price: productData.price || 0,
            stock: productData.stock || 0,
            images: productData.images || []
          },
          update: {
            price: productData.price || 0,
            stock: productData.stock || 0,
            images: productData.images || []
          }
        });
      }

      // সিডিউল আপডেট করুন
      const nextImportTime = new Date();
      nextImportTime.setHours(nextImportTime.getHours() + schedule.importInterval);

      await this.prisma.aPIImportSchedule.update({
        where: { id: schedule.id },
        data: {
          lastImportedAt: new Date(),
          nextImportAt: nextImportTime
        }
      });

      this.logger.log(
        `Imported ${products.length} products for tenant ${schedule.tenantId}`
      );
    } catch (err) {
      this.logger.error('API import failed:', err);
      throw err;
    }
  }

  // API সিডিউল সেট করুন
  async createAPIImportSchedule(
    tenantId: string,
    apiUrl: string,
    apiKey?: string,
    importInterval: number = 6
  ) {
    const schedule = await this.prisma.aPIImportSchedule.create({
      data: {
        tenantId,
        apiUrl,
        apiKey,
        importInterval
      }
    });

    return { success: true, data: schedule };
  }
}