import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import * as AWS from 'aws-sdk';
import * as fs from 'fs';

@Injectable()
export class BackupAutomationService {
  private logger = new Logger('BackupAutomationService');
  private s3 = new AWS.S3();

  constructor(private prisma: PrismaService) {}

  // প্রতিদিন মধ্যরাতে ব্যাকআপ করুন
  @Cron('0 0 * * *')
  async performDailyBackup() {
    this.logger.log('Starting daily backups...');

    const tenants = await this.prisma.tenant.findMany();

    for (const tenant of tenants) {
      try {
        await this.backupTenantData(tenant.id);
      } catch (err) {
        this.logger.error(`Backup failed for tenant ${tenant.id}:`, err);
      }
    }
  }

  private async backupTenantData(tenantId: string) {
    // টেন্যান্ট ডেটা এক্সপোর্ট করুন
    const orders = await this.prisma.order.findMany({
      where: { tenantId },
      take: 10000
    });

    const products = await this.prisma.product.findMany({
      where: { tenantId },
      take: 10000
    });

    const backupData = {
      tenantId,
      exportDate: new Date().toISOString(),
      orders,
      products
    };

    const backupJson = JSON.stringify(backupData, null, 2);
    const fileName = `backup-${tenantId}-${Date.now()}.json`;

    // S3-এ আপলোড করুন
    const key = `backups/${tenantId}/${fileName}`;

    await this.s3
      .putObject({
        Bucket: process.env.AWS_S3_BUCKET || '',
        Key: key,
        Body: backupJson,
        ContentType: 'application/json'
      })
      .promise();

    // ব্যাকআপ লগ সংরক্ষণ করুন
    await this.prisma.backupLog.create({
      data: {
        tenantId,
        backupType: 'full',
        backupUrl: `https://${process.env.AWS_S3_BUCKET}.s3.amazonaws.com/${key}`,
        backupSize: BigInt(backupJson.length),
        status: 'success'
      }
    });

    this.logger.log(`Backup completed for tenant ${tenantId}`);
  }

  // ম্যানুয়াল ব্যাকআপ ট্রিগার করুন
  async triggerManualBackup(tenantId: string) {
    await this.backupTenantData(tenantId);
    return { success: true };
  }
}