import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import * as nodemailer from 'nodemailer';

@Injectable()
export class CartRecoveryMultiChannelService {
  private logger = new Logger('CartRecoveryMultiChannelService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // প্রতি ৩০ মিনিটে পরিত্যক্ত কার্ট রিকভারি চেষ্টা করুন
  @Cron('*/30 * * * *')
  async recoverAbandonedCarts() {
    this.logger.log('Running cart recovery automation...');

    const thirtyMinutesAgo = new Date(Date.now() - 30 * 60 * 1000);

    const abandonedCarts = await this.prisma.abandonedCart.findMany({
      where: {
        createdAt: { gt: thirtyMinutesAgo },
        reminderSent: false,
        recoveredAt: null
      },
      include: { user: true }
    });

    for (const cart of abandonedCarts) {
      try {
        // মাল্টি-চ্যানেল রিকভারি চেষ্টা করুন
        if (cart.user.email) {
          await this.sendEmailRecovery(cart);
        }

        if (cart.user.phone) {
          await this.sendSMSRecovery(cart);
        }

        // WhatsApp (যদি নম্বর থাকে)
        if (cart.user.phone && cart.user.whatsappOptIn) {
          await this.sendWhatsAppRecovery(cart);
        }

        await this.prisma.abandonedCart.update({
          where: { id: cart.id },
          data: { reminderSent: true, reminderSentAt: new Date() }
        });
      } catch (err) {
        this.logger.error(`Cart recovery failed: ${cart.id}:`, err);
      }
    }
  }

  private async sendEmailRecovery(cart: any) {
    const cartItems = cart.cartItems as any[];
    const recoveryUrl = `${process.env.APP_URL}/cart/recover/${cart.id}`;

    const emailHtml = `
<h2>Your Cart is Still Waiting!</h2>
<p>Hi ${cart.user.name},</p>
<p>You left ${cartItems.length} item(s) in your cart. Complete your purchase now and get 10% off!</p>

${cartItems.map((item) => `<p>${item.productName} x${item.quantity}</p>`).join('')}

<p><strong>Total: $${cart.cartTotal}</strong></p>
<p><a href="${recoveryUrl}">Complete Purchase</a></p>
<p>This offer expires in 24 hours!</p>
    `;

    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL || 'noreply@store.com',
      to: cart.user.email,
      subject: `Don't miss out! You left ${cartItems.length} items`,
      html: emailHtml
    });
  }

  private async sendSMSRecovery(cart: any) {
    const message = `You left ${(cart.cartItems as any[]).length} items in your cart! Complete your purchase and get 10% off: ${process.env.APP_URL}/cart/recover/${cart.id}`;

    // Twilio বা অন্যান্য SMS সেবা ব্যবহার করুন
    this.logger.log(`SMS sent to ${cart.user.phone}: ${message}`);
  }

  private async sendWhatsAppRecovery(cart: any) {
    const message = `Hey ${cart.user.name}! Your cart is waiting with ${(cart.cartItems as any[]).length} items. Get 10% off if you complete your purchase now!`;

    // Meta WhatsApp API বা Twilio
    this.logger.log(`WhatsApp sent to ${cart.user.phone}`);
  }
}