import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import * as nodemailer from 'nodemailer';

@Injectable()
export class CartRecoveryAutomationService {
  private logger = new Logger('CartRecoveryAutomationService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // প্রতি ঘণ্টায় পরিত্যক্ত কার্ট চেক করুন
  @Cron('0 * * * *')
  async checkAbandonedCarts() {
    this.logger.log('Checking for abandoned carts...');

    // 1 ঘণ্টা আগে তৈরি পরিত্যক্ত কার্ট পান
    const oneHourAgo = new Date(Date.now() - 60 * 60 * 1000);

    const abandonedCarts = await this.prisma.abandonedCart.findMany({
      where: {
        reminderSent: false,
        createdAt: { lt: oneHourAgo }
      },
      include: { user: true }
    });

    for (const cart of abandonedCarts) {
      try {
        await this.sendRecoveryEmail(cart);
      } catch (err) {
        this.logger.error(`Cart recovery email failed for cart ${cart.id}:`, err);
      }
    }
  }

  private async sendRecoveryEmail(cart: any) {
    if (!cart.user.email) return;

    const cartItems = cart.cartItems as any[];
    const recoveryLink = `${process.env.APP_URL}/cart/recover/${cart.id}`;

    const itemsHtml = cartItems
      .map(
        (item) => `
      <div style="border: 1px solid #ddd; padding: 10px; margin: 10px 0;">
        <p><strong>${item.productName}</strong></p>
        <p>Quantity: ${item.quantity}</p>
        <p>Price: $${item.price}</p>
      </div>
    `
      )
      .join('');

    const emailHtml = `
<h2>You left something in your cart!</h2>
<p>Hi ${cart.user.name},</p>
<p>We noticed you left your cart with ${cartItems.length} item(s). Complete your order now!</p>

<div>
  ${itemsHtml}
</div>

<p><strong>Cart Total: $${cart.cartTotal}</strong></p>

<p><a href="${recoveryLink}" style="background: #007bff; color: white; padding: 12px 20px; text-decoration: none; border-radius: 4px; display: inline-block;">Complete Your Purchase</a></p>

<p>This offer expires in 24 hours.</p>
    `;

    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL || 'noreply@store.com',
      to: cart.user.email,
      subject: `You left items in your cart!`,
      html: emailHtml
    });

    // কার্ট রিমাইন্ডার মার্ক করুন
    await this.prisma.abandonedCart.update({
      where: { id: cart.id },
      data: { reminderSent: true, reminderSentAt: new Date() }
    });

    this.logger.log(`Cart recovery email sent for cart ${cart.id}`);
  }

  // কার্ট ত্যাগ ট্র্যাক করুন
  async trackAbandonedCart(userId: string, cartItems: any[], cartTotal: number) {
    const cart = await this.prisma.abandonedCart.create({
      data: {
        userId,
        cartItems,
        cartTotal
      }
    });

    return { success: true };
  }
}