import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import * as nodemailer from 'nodemailer';

@Injectable()
export class DailyReportAutomationService {
  private logger = new Logger('DailyReportAutomationService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // প্রতিদিন সকাল ৮টায় রিপোর্ট পাঠান
  @Cron('0 8 * * *')
  async sendDailyReports() {
    this.logger.log('Sending daily reports to all admins...');

    const tenants = await this.prisma.tenant.findMany({
      where: { type: 'admin' }
    });

    for (const tenant of tenants) {
      try {
        const report = await this.generateDailyReport(tenant.id);
        await this.sendReportEmail(tenant.email, report);
      } catch (err) {
        this.logger.error(`Daily report failed for tenant ${tenant.id}:`, err);
      }
    }
  }

  private async generateDailyReport(tenantId: string) {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const tomorrow = new Date(today);
    tomorrow.setDate(tomorrow.getDate() + 1);

    // অর্ডার ডেটা
    const orders = await this.prisma.order.findMany({
      where: {
        tenantId,
        createdAt: { gte: today, lt: tomorrow }
      }
    });

    // নতুন ব্যবহারকারী
    const newUsers = await this.prisma.user.count({
      where: {
        createdAt: { gte: today, lt: tomorrow }
      }
    });

    // শীর্ষ পণ্য
    const topProducts = await this.prisma.product.findMany({
      where: { tenantId },
      orderBy: { views: 'desc' },
      take: 5
    });

    const totalRevenue = orders.reduce((sum, o) => sum + o.total, 0);

    return {
      date: today.toLocaleDateString(),
      orders: orders.length,
      revenue: totalRevenue,
      newUsers,
      topProducts: topProducts.map((p) => ({ name: p.title, views: p.views }))
    };
  }

  private async sendReportEmail(email: string, report: any) {
    const emailHtml = `
<h2>Daily Store Report - ${report.date}</h2>

<table style="border-collapse: collapse; width: 100%;">
  <tr style="background: #f0f0f0;">
    <th style="border: 1px solid #ddd; padding: 10px;">Metric</th>
    <th style="border: 1px solid #ddd; padding: 10px;">Value</th>
  </tr>
  <tr>
    <td style="border: 1px solid #ddd; padding: 10px;">Total Orders</td>
    <td style="border: 1px solid #ddd; padding: 10px;">${report.orders}</td>
  </tr>
  <tr>
    <td style="border: 1px solid #ddd; padding: 10px;">Total Revenue</td>
    <td style="border: 1px solid #ddd; padding: 10px;">$${report.revenue.toFixed(2)}</td>
  </tr>
  <tr>
    <td style="border: 1px solid #ddd; padding: 10px;">New Users</td>
    <td style="border: 1px solid #ddd; padding: 10px;">${report.newUsers}</td>
  </tr>
</table>

<h3>Top Products</h3>
<ul>
${report.topProducts.map((p: any) => `<li>${p.name} - ${p.views} views</li>`).join('')}
</ul>
    `;

    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL || 'noreply@store.com',
      to: email,
      subject: `Daily Report - ${report.date}`,
      html: emailHtml
    });
  }
}