import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import * as nodemailer from 'nodemailer';

@Injectable()
export class CustomerFollowupService {
  private logger = new Logger('CustomerFollowupService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // প্রতি মিনিটে ফলো-আপ মেসেজ পাঠান
  @Cron('* * * * *')
  async sendScheduledFollowups() {
    const now = new Date();

    const sequences = await this.prisma.followUpSequence.findMany({
      where: {
        scheduledFor: { lte: now },
        sentAt: null
      },
      include: { user: true }
    });

    for (const sequence of sequences) {
      try {
        if (sequence.stepType === 'email') {
          await this.sendFollowupEmail(sequence);
        } else if (sequence.stepType === 'sms') {
          await this.sendFollowupSMS(sequence);
        } else if (sequence.stepType === 'whatsapp') {
          await this.sendFollowupWhatsApp(sequence);
        }

        await this.prisma.followUpSequence.update({
          where: { id: sequence.id },
          data: { sentAt: new Date() }
        });

        this.logger.log(`Follow-up sent: ${sequence.id}`);
      } catch (err) {
        this.logger.error(`Follow-up failed: ${sequence.id}:`, err);
      }
    }
  }

  private async sendFollowupEmail(sequence: any) {
    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL || 'noreply@store.com',
      to: sequence.user.email,
      subject: 'We miss you!',
      html: sequence.message
    });
  }

  private async sendFollowupSMS(sequence: any) {
    // Twilio বা অন্যান্য SMS প্রোভাইডার ব্যবহার করুন
    this.logger.log(`SMS sent to ${sequence.user.phone}: ${sequence.message}`);
  }

  private async sendFollowupWhatsApp(sequence: any) {
    // WhatsApp API (Twilio, Meta, ইত্যাদি)
    this.logger.log(`WhatsApp sent to ${sequence.user.phone}`);
  }

  // ফলো-আপ সিকোয়েন্স তৈরি করুন
  async createFollowupSequence(userId: string, sequenceType: string) {
    const templates: any = {
      post_purchase: [
        {
          step: 1,
          delay: 24, // ২৪ ঘণ্টা পর
          type: 'email',
          message: 'Thank you for your purchase! How was your experience?'
        },
        {
          step: 2,
          delay: 72, // ৩ দিন পর
          type: 'email',
          message: 'We noticed you purchased [product]. Would you like to leave a review?'
        },
        {
          step: 3,
          delay: 168, // ১ সপ্তাহ পর
          type: 'email',
          message: 'Enjoy [discount]% off your next purchase!'
        }
      ],
      abandoned_cart: [
        {
          step: 1,
          delay: 1, // ১ ঘণ্টা পর
          type: 'email',
          message: 'You left items in your cart. Complete your purchase now!'
        },
        {
          step: 2,
          delay: 24, // ২৪ ঘণ্টা পর
          type: 'sms',
          message: 'Final reminder: Your cart is waiting. Use code SAVE10 for 10% off'
        }
      ]
    };

    const template = templates[sequenceType] || [];

    for (const step of template) {
      const scheduledFor = new Date();
      scheduledFor.setHours(scheduledFor.getHours() + step.delay);

      await this.prisma.followUpSequence.create({
        data: {
          userId,
          sequenceType,
          step: step.step,
          stepType: step.type,
          message: step.message,
          scheduledFor
        }
      });
    }

    return { success: true, sequenceCount: template.length };
  }
}