import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as PDFDocument from 'pdfkit';
import * as AWS from 'aws-sdk';
import * as nodemailer from 'nodemailer';

@Injectable()
export class InvoiceAutomationService {
  private logger = new Logger('InvoiceAutomationService');
  private s3 = new AWS.S3();
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // অর্ডার কনফার্ম হলে ইনভয়েস তৈরি করুন
  async generateAndSendInvoice(orderId: string) {
    try {
      const order = await this.prisma.order.findUnique({
        where: { id: orderId },
        include: {
          items: { include: { product: true } },
          user: true
        }
      });

      if (!order || !order.user) throw new Error('Order not found');

      // ইনভয়েস জেনারেট করুন
      const pdfPath = await this.generateInvoicePDF(order);

      // S3-এ আপলোড করুন
      const s3Url = await this.uploadInvoiceToS3(pdfPath, orderId);

      // ইনভয়েস অ্যাডভ অ্যার্কাইভ করুন
      const invoice = await this.prisma.invoiceArchive.create({
        data: {
          orderId,
          invoiceNumber: order.orderNumber,
          invoicePdf: s3Url
        }
      });

      // ইমেইল পাঠান
      await this.sendInvoiceEmail(order.user.email, s3Url, order.orderNumber);

      this.logger.log(`Invoice generated and sent for order ${orderId}`);

      return { success: true, invoiceUrl: s3Url };
    } catch (err) {
      this.logger.error('Invoice generation failed:', err);
      throw err;
    }
  }

  private async generateInvoicePDF(order: any): Promise<string> {
    const doc = new PDFDocument();
    const filename = `/tmp/invoice-${order.orderNumber}.pdf`;

    // হেডার
    doc.fontSize(20).text('INVOICE', 100, 100);
    doc.fontSize(10).text(`Invoice #: ${order.orderNumber}`);
    doc.text(`Date: ${new Date().toLocaleDateString()}`);

    // কাস্টমার ইনফো
    doc.fontSize(12).text(`Customer: ${order.user.name}`);
    doc.text(`Email: ${order.user.email}`);

    // আইটেম
    doc.fontSize(10).text('Items:', 100, 250);
    let yPosition = 270;

    for (const item of order.items) {
      doc.text(`${item.product.title} x${item.quantity} = $${item.price * item.quantity}`);
      yPosition += 20;
    }

    // টোটাল
    doc.fontSize(12).text(`Total: $${order.total}`, 100, yPosition + 20);

    // ফাইলে সংরক্ষণ করুন
    doc.pipe(require('fs').createWriteStream(filename));
    doc.end();

    return filename;
  }

  private async uploadInvoiceToS3(pdfPath: string, orderId: string): Promise<string> {
    const fileContent = require('fs').readFileSync(pdfPath);
    const key = `invoices/${orderId}/${Date.now()}.pdf`;

    await this.s3
      .putObject({
        Bucket: process.env.AWS_S3_BUCKET || '',
        Key: key,
        Body: fileContent,
        ContentType: 'application/pdf'
      })
      .promise();

    return `https://${process.env.AWS_S3_BUCKET}.s3.amazonaws.com/${key}`;
  }

  private async sendInvoiceEmail(
    email: string,
    invoiceUrl: string,
    orderNumber: string
  ) {
    const emailHtml = `
<h2>Your Invoice</h2>
<p>Thank you for your order!</p>
<p>Order Number: ${orderNumber}</p>
<p><a href="${invoiceUrl}">Download Invoice</a></p>
    `;

    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL || 'noreply@store.com',
      to: email,
      subject: `Invoice - Order ${orderNumber}`,
      html: emailHtml,
      attachments: [
        {
          filename: `invoice-${orderNumber}.pdf`,
          path: invoiceUrl
        }
      ]
    });
  }
}