import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';

@Injectable()
export class N8NService {
  private logger = new Logger('N8NService');
  private n8nBaseUrl = process.env.N8N_URL || 'http://localhost:5678';
  private n8nApiKey = process.env.N8N_API_KEY || '';

  constructor(private prisma: PrismaService) {}

  // N8N ওয়ার্কফ্লো তৈরি করুন
  async createWorkflow(tenantId: string, workflowConfig: any) {
    try {
      const response = await axios.post(
        `${this.n8nBaseUrl}/api/v1/workflows`,
        {
          name: workflowConfig.name,
          active: true,
          nodes: workflowConfig.nodes,
          connections: workflowConfig.connections,
          settings: workflowConfig.settings
        },
        {
          headers: {
            'X-N8N-API-KEY': this.n8nApiKey
          }
        }
      );

      const workflowId = response.data.id;

      // ওয়ার্কফ্লো ডাটাবেসে সংরক্ষণ করুন
      const workflow = await this.prisma.automationWorkflow.create({
        data: {
          tenantId,
          name: workflowConfig.name,
          description: workflowConfig.description,
          type: 'n8n',
          trigger: workflowConfig.trigger,
          action: workflowConfig.action,
          config: {
            n8nWorkflowId: workflowId,
            ...workflowConfig
          }
        }
      });

      this.logger.log(`N8N workflow created: ${workflow.id}`);

      return { success: true, data: workflow };
    } catch (err) {
      this.logger.error('Workflow creation failed:', err);
      throw err;
    }
  }

  // ওয়ার্কফ্লো চালান
  async executeWorkflow(workflowId: string, data?: any) {
    try {
      const workflow = await this.prisma.automationWorkflow.findUnique({
        where: { id: workflowId }
      });

      if (!workflow) throw new Error('Workflow not found');

      const n8nWorkflowId = workflow.config?.n8nWorkflowId;

      const response = await axios.post(
        `${this.n8nBaseUrl}/api/v1/workflows/${n8nWorkflowId}/execute`,
        { data },
        {
          headers: {
            'X-N8N-API-KEY': this.n8nApiKey
          }
        }
      );

      // এক্সিকিউশন লগ সংরক্ষণ করুন
      await this.prisma.automationLog.create({
        data: {
          workflowId,
          status: 'success',
          input: data,
          output: response.data
        }
      });

      return { success: true, data: response.data };
    } catch (err) {
      this.logger.error('Workflow execution failed:', err);

      // ফেইলিউর লগ করুন
      await this.prisma.automationLog.create({
        data: {
          workflowId,
          status: 'failed',
          error: (err as Error).message
        }
      });

      throw err;
    }
  }

  // প্রি-বিল্ট ওয়ার্কফ্লো টেমপ্লেট
  getWorkflowTemplates() {
    return {
      templates: [
        {
          id: 'order_confirmation',
          name: 'Order Confirmation Email',
          description: 'Send confirmation email after order placement',
          trigger: 'order_placed',
          action: 'send_email',
          nodes: [
            {
              name: 'Trigger',
              type: 'webhook',
              typeVersion: 1,
              position: [250, 300],
              parameters: {
                httpMethod: 'POST',
                path: 'order-webhook'
              }
            },
            {
              name: 'Send Email',
              type: 'email',
              typeVersion: 1,
              position: [450, 300],
              parameters: {
                fromEmail: 'noreply@store.com',
                toEmail: '={{$json.customerEmail}}',
                subject: 'Order Confirmation #{{$json.orderNumber}}',
                textHtml: '<h1>Thank you for your order!</h1>'
              }
            }
          ],
          connections: {
            Trigger: {
              main: [[{ node: 'Send Email', type: 'main', index: 0 }]]
            }
          }
        },

        {
          id: 'low_stock_alert',
          name: 'Low Stock Alert',
          description: 'Alert when product stock is low',
          trigger: 'low_stock',
          action: 'send_notification',
          nodes: [
            {
              name: 'Check Stock',
              type: 'schedule',
              typeVersion: 1,
              position: [250, 300],
              parameters: {
                rule: {
                  interval: [1],
                  intervalUnit: 'hours'
                }
              }
            },
            {
              name: 'Send Alert',
              type: 'notification',
              typeVersion: 1,
              position: [450, 300]
            }
          ]
        },

        {
          id: 'price_update',
          name: 'Auto Price Update',
          description: 'Update product prices based on rules',
          trigger: 'schedule',
          action: 'update_price'
        },

        {
          id: 'affiliate_payout',
          name: 'Affiliate Auto Payout',
          description: 'Automatically process affiliate commissions',
          trigger: 'schedule',
          action: 'payout'
        }
      ]
    };
  }
}