import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as nodemailer from 'nodemailer';

@Injectable()
export class OrderEmailAutomationService {
  private logger = new Logger('OrderEmailAutomationService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // অর্ডার তৈরি হওয়ার সময় স্বয়ংক্রিয় ইমেইল পাঠান
  async sendOrderConfirmationEmail(orderId: string) {
    try {
      const order = await this.prisma.order.findUnique({
        where: { id: orderId },
        include: {
          items: { include: { product: true } },
          user: true,
          shippingAddress: true
        }
      });

      if (!order || !order.user) throw new Error('Order not found');

      // ইমেইল টেমপ্লেট তৈরি করুন
      const emailHtml = this.generateOrderConfirmationHtml(order);

      // ইমেইল পাঠান
      await this.mailer.sendMail({
        from: process.env.STORE_EMAIL || 'noreply@store.com',
        to: order.user.email,
        subject: `Order Confirmation - #${order.orderNumber}`,
        html: emailHtml
      });

      this.logger.log(`Order confirmation email sent for order ${orderId}`);

      // ইভেন্ট লগ করুন
      await this.prisma.automationLog.create({
        data: {
          workflowId: 'order_confirmation',
          status: 'success',
          input: { orderId }
        }
      });

      return { success: true };
    } catch (err) {
      this.logger.error('Order email send failed:', err);

      await this.prisma.automationLog.create({
        data: {
          workflowId: 'order_confirmation',
          status: 'failed',
          error: (err as Error).message
        }
      });

      throw err;
    }
  }

  private generateOrderConfirmationHtml(order: any): string {
    const trackingUrl = `${process.env.APP_URL}/orders/${order.id}/tracking`;

    return `
<!DOCTYPE html>
<html>
<head>
  <style>
    body { font-family: Arial, sans-serif; color: #333; }
    .container { max-width: 600px; margin: 0 auto; padding: 20px; }
    .header { background: #007bff; color: white; padding: 20px; border-radius: 4px; }
    .order-items { margin: 20px 0; }
    .item { border: 1px solid #ddd; padding: 10px; margin: 10px 0; }
    .button { background: #007bff; color: white; padding: 12px 20px; text-decoration: none; border-radius: 4px; display: inline-block; }
  </style>
</head>
<body>
  <div class="container">
    <div class="header">
      <h1>Order Confirmation</h1>
      <p>Order #${order.orderNumber}</p>
    </div>

    <p>Hi ${order.user.name},</p>
    <p>Thank you for your order! We're preparing it for shipment.</p>

    <div class="order-items">
      <h3>Order Items:</h3>
      ${order.items
        .map(
          (item: any) => `
        <div class="item">
          <p><strong>${item.product.title}</strong></p>
          <p>Quantity: ${item.quantity}</p>
          <p>Price: $${item.price}</p>
        </div>
      `
        )
        .join('')}
    </div>

    <p><strong>Total Amount:</strong> $${order.total}</p>

    <p><strong>Shipping Address:</strong><br/>
    ${order.shippingAddress.address}<br/>
    ${order.shippingAddress.city}, ${order.shippingAddress.zipCode}</p>

    <p><a href="${trackingUrl}" class="button">Track Your Order</a></p>

    <p>Thank you for shopping with us!</p>
  </div>
</body>
</html>
    `;
  }

  // শিপমেন্ট আপডেট ইমেইল পাঠান
  async sendShipmentUpdateEmail(orderId: string, trackingNumber: string) {
    const order = await this.prisma.order.findUnique({
      where: { id: orderId },
      include: { user: true }
    });

    if (!order || !order.user) return;

    const trackingUrl = `${process.env.APP_URL}/orders/${orderId}/tracking`;

    const emailHtml = `
<h2>Your Order Has Been Shipped!</h2>
<p>Order #${order.orderNumber}</p>
<p>Tracking Number: ${trackingNumber}</p>
<p><a href="${trackingUrl}">Track your order</a></p>
    `;

    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL || 'noreply@store.com',
      to: order.user.email,
      subject: `Your Order #${order.orderNumber} Has Shipped`,
      html: emailHtml
    });
  }
}