import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';

@Injectable()
export class OrderPushSupplierService {
  private logger = new Logger('OrderPushSupplierService');

  constructor(private prisma: PrismaService) {}

  // নতুন অর্ডার সরবরাহকারীকে পাঠান
  async pushOrderToSupplier(orderId: string, supplierId: string) {
    try {
      const order = await this.prisma.order.findUnique({
        where: { id: orderId },
        include: { items: { include: { product: true } } }
      });

      if (!order) throw new Error('Order not found');

      // সরবরাহকারীর API এন্ডপয়েন্ট খুঁজুন
      const supplier = await this.prisma.wholesaler.findUnique({
        where: { id: supplierId }
      });

      if (!supplier || !supplier.apiUrl) {
        // ফেলব্যাক: অর্ডার কিউতে যোগ করুন
        await this.prisma.supplierOrderQueue.create({
          data: {
            orderId,
            supplierId,
            status: 'pending'
          }
        });

        return { success: true, queued: true };
      }

      // সরবরাহকারীর API-তে অর্ডার পাঠান
      const payload = {
        orderId: order.orderNumber,
        items: order.items.map((item) => ({
          productSku: item.product.sku,
          quantity: item.quantity
        })),
        shippingAddress: order.shippingAddress,
        customer: {
          name: order.user?.name || 'Unknown',
          email: order.user?.email || ''
        }
      };

      await axios.post(`${supplier.apiUrl}/orders`, payload, {
        headers: {
          'Authorization': `Bearer ${supplier.apiKey}`,
          'Content-Type': 'application/json'
        }
      });

      // অর্ডার কিউ থেকে মার্ক করুন
      await this.prisma.supplierOrderQueue.create({
        data: {
          orderId,
          supplierId,
          status: 'sent',
          sentAt: new Date()
        }
      });

      this.logger.log(`Order ${orderId} pushed to supplier ${supplierId}`);

      return { success: true, sent: true };
    } catch (err) {
      this.logger.error('Order push failed:', err);

      // কিউতে রাখুন পরে পাঠানোর জন্য
      await this.prisma.supplierOrderQueue.create({
        data: {
          orderId,
          supplierId,
          status: 'pending'
        }
      });

      return { success: false, queued: true };
    }
  }

  // পেন্ডিং অর্ডার পুনরায় চেষ্টা করুন
  async retryPendingOrders() {
    const pendingOrders = await this.prisma.supplierOrderQueue.findMany({
      where: { status: 'pending' }
    });

    for (const queue of pendingOrders) {
      try {
        await this.pushOrderToSupplier(queue.orderId, queue.supplierId);
      } catch (err) {
        this.logger.warn(`Retry failed for order ${queue.orderId}`);
      }
    }
  }
}