import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import * as nodemailer from 'nodemailer';

@Injectable()
export class PerformanceMonitorService {
  private logger = new Logger('PerformanceMonitorService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // প্রতি ঘণ্টায় পারফরম্যান্স চেক করুন
  @Cron('0 * * * *')
  async monitorPerformance() {
    this.logger.log('Monitoring platform performance...');

    const tenants = await this.prisma.tenant.findMany({
      where: { type: 'admin' }
    });

    for (const tenant of tenants) {
      try {
        await this.checkTenantPerformance(tenant.id, tenant.email);
      } catch (err) {
        this.logger.error(`Performance check failed for tenant ${tenant.id}:`, err);
      }
    }
  }

  private async checkTenantPerformance(tenantId: string, adminEmail: string) {
    // সেলস মেট্রিক্স সংগ্রহ করুন
    const lastHour = new Date(Date.now() - 60 * 60 * 1000);

    const recentOrders = await this.prisma.order.count({
      where: {
        tenantId,
        createdAt: { gte: lastHour }
      }
    });

    const previousHourOrders = await this.prisma.order.count({
      where: {
        tenantId,
        createdAt: {
          gte: new Date(Date.now() - 2 * 60 * 60 * 1000),
          lt: lastHour
        }
      }
    });

    // ড্রপ ডিটেক্ট করুন
    const dropPercentage =
      previousHourOrders > 0
        ? ((previousHourOrders - recentOrders) / previousHourOrders) * 100
        : 0;

    if (dropPercentage > 30) {
      // 30% এরও বেশি ড্রপ
      const alert = await this.createAlert(
        tenantId,
        'sales_drop',
        'critical',
        `Sales dropped by ${dropPercentage.toFixed(2)}%`
      );

      await this.sendAlertEmail(
        adminEmail,
        'CRITICAL: Sales Performance Drop Detected',
        `Your store sales dropped by ${dropPercentage.toFixed(2)}% in the last hour.`
      );
    }

    // ট্রাফিক মেট্রিক্স (API কল কাউন্ট)
    const apiCalls = await this.countAPICallsLastHour(tenantId);

    if (apiCalls < 100 && apiCalls > 0) {
      // ট্রাফিক খুব কম
      const alert = await this.createAlert(
        tenantId,
        'traffic_drop',
        'high',
        `Traffic significantly reduced: ${apiCalls} API calls`
      );

      await this.sendAlertEmail(
        adminEmail,
        'WARNING: Traffic Drop Detected',
        `Your store traffic has dropped significantly.`
      );
    }

    // মেট্রিক্স রেকর্ড করুন
    await this.prisma.performanceMetric.create({
      data: {
        tenantId,
        metric: 'sales',
        value: recentOrders,
        status: dropPercentage > 30 ? 'critical' : 'normal'
      }
    });
  }

  private async createAlert(
    tenantId: string,
    alertType: string,
    severity: string,
    message: string
  ) {
    return await this.prisma.performanceAlert.create({
      data: {
        tenantId,
        alertType,
        severity,
        message
      }
    });
  }

  private async sendAlertEmail(to: string, subject: string, message: string) {
    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL || 'noreply@store.com',
      to,
      subject,
      html: `<p>${message}</p>`
    });
  }

  private async countAPICallsLastHour(tenantId: string): Promise<number> {
    // TODO: Implement API call logging
    return 0;
  }

  // অ্যালার্ট সমাধান করুন
  async resolveAlert(alertId: string) {
    await this.prisma.performanceAlert.update({
      where: { id: alertId },
      data: {
        isResolved: true,
        resolvedAt: new Date()
      }
    });

    return { success: true };
  }
}