import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class WholesalerPriceSyncService {
  private logger = new Logger('WholesalerPriceSyncService');

  constructor(private prisma: PrismaService) {}

  // হোলসেলার দাম পরিবর্তন করলে সংযুক্ত স্টোরে সিঙ্ক করুন
  async syncPriceToConnectedStores(wholesalerId: string, productId: string, newPrice: number) {
    try {
      // পণ্য পান
      const product = await this.prisma.product.findUnique({
        where: { id: productId }
      });

      if (!product) throw new Error('Product not found');

      // সংযুক্ত অ্যাডমিন খুঁজুন
      const connections = await this.prisma.adminWholesalerConnection.findMany({
        where: { wholesalerTenantId: wholesalerId, status: 'active' }
      });

      // প্রতিটি সংযুক্ত স্টোরে দাম আপডেট করুন
      for (const connection of connections) {
        const adminProduct = await this.prisma.product.findFirst({
          where: {
            tenantId: connection.adminTenantId,
            sku: product.sku
          }
        });

        if (adminProduct && connection.autoSyncPrices) {
          await this.prisma.product.update({
            where: { id: adminProduct.id },
            data: { price: newPrice }
          });

          this.logger.log(
            `Price synced for product ${productId} to store ${connection.adminTenantId}`
          );
        }
      }

      return { success: true, syncedStores: connections.length };
    } catch (err) {
      this.logger.error('Price sync failed:', err);
      throw err;
    }
  }
}