import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';

@Injectable()
export class PriceUpdateAutomationService {
  private logger = new Logger('PriceUpdateAutomationService');

  constructor(private prisma: PrismaService) {}

  // প্রতিদিন নির্ধারিত সময়ে মূল্য আপডেট করুন
  @Cron('0 2 * * *') // প্রতিদিন সকাল 2টায়
  async autoUpdatePrices() {
    this.logger.log('Starting automatic price updates...');

    const workflows = await this.prisma.automationWorkflow.findMany({
      where: {
        action: 'update_price',
        isActive: true
      }
    });

    for (const workflow of workflows) {
      try {
        await this.updatePricesForTenant(workflow.tenantId, workflow.config);

        await this.prisma.automationLog.create({
          data: {
            workflowId: workflow.id,
            status: 'success'
          }
        });
      } catch (err) {
        this.logger.error(`Price update failed for tenant ${workflow.tenantId}:`, err);

        await this.prisma.automationLog.create({
          data: {
            workflowId: workflow.id,
            status: 'failed',
            error: (err as Error).message
          }
        });
      }
    }
  }

  private async updatePricesForTenant(tenantId: string, config: any) {
    const rules = config.priceRules || [];

    for (const rule of rules) {
      // কন্ডিশন চেক করুন
      if (rule.condition === 'time_based') {
        // সময়-ভিত্তিক মূল্য আপডেট (সিজনাল)
        const hour = new Date().getHours();

        if (rule.startHour <= hour && hour < rule.endHour) {
          await this.applyPriceChange(tenantId, rule);
        }
      } else if (rule.condition === 'inventory_based') {
        // ইনভেন্টরি-ভিত্তিক মূল্য আপডেট
        const lowStockProducts = await this.prisma.product.findMany({
          where: {
            tenantId,
            stock: { lt: rule.lowStockThreshold }
          }
        });

        for (const product of lowStockProducts) {
          const newPrice = product.price * (1 - rule.discountPercent / 100);
          await this.prisma.product.update({
            where: { id: product.id },
            data: { price: newPrice }
          });
        }
      } else if (rule.condition === 'competitor_based') {
        // প্রতিযোগী-ভিত্তিক মূল্য (API থেকে পান)
        // TODO: Implement competitor price tracking
      }
    }
  }

  private async applyPriceChange(tenantId: string, rule: any) {
    const products = await this.prisma.product.findMany({
      where: { tenantId, categoryId: rule.categoryId }
    });

    for (const product of products) {
      let newPrice = product.price;

      if (rule.adjustmentType === 'percentage') {
        newPrice = product.price * (1 + rule.adjustmentValue / 100);
      } else if (rule.adjustmentType === 'fixed') {
        newPrice = rule.adjustmentValue;
      }

      await this.prisma.product.update({
        where: { id: product.id },
        data: { price: newPrice }
      });
    }
  }

  // মূল্য আপডেট রুল তৈরি করুন
  async createPriceRule(tenantId: string, rule: any) {
    const workflow = await this.prisma.automationWorkflow.create({
      data: {
        tenantId,
        name: `Price Rule - ${rule.name}`,
        type: 'cron',
        trigger: 'schedule',
        action: 'update_price',
        config: { priceRules: [rule] },
        isActive: true
      }
    });

    return { success: true, data: workflow };
  }
}