import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import axios from 'axios';

@Injectable()
export class ProductSyncAutomationService {
  private logger = new Logger('ProductSyncAutomationService');

  constructor(private prisma: PrismaService) {}

  // স্বয়ংক্রিয় পণ্য সিঙ্ক (প্রতি ৬ ঘণ্টায়)
  @Cron('0 */6 * * *')
  async autoSyncProducts() {
    this.logger.log('Starting automatic product sync...');

    const workflows = await this.prisma.automationWorkflow.findMany({
      where: {
        type: 'cron',
        action: 'sync_product',
        isActive: true
      }
    });

    for (const workflow of workflows) {
      try {
        await this.syncProductsForTenant(workflow.tenantId, workflow.config);
      } catch (err) {
        this.logger.error(`Sync failed for tenant ${workflow.tenantId}:`, err);

        await this.prisma.automationLog.create({
          data: {
            workflowId: workflow.id,
            status: 'failed',
            error: (err as Error).message
          }
        });
      }
    }
  }

  private async syncProductsForTenant(tenantId: string, config: any) {
    // এক্সটার্নাল সোর্স থেকে পণ্য আনুন
    const sourceUrl = config.sourceUrl || '';
    const sourceType = config.sourceType || 'csv'; // csv, json, api

    let products = [];

    if (sourceType === 'api') {
      const response = await axios.get(sourceUrl, {
        headers: config.headers || {}
      });
      products = response.data.products || [];
    } else if (sourceType === 'csv') {
      // CSV থেকে পণ্য পার্স করুন
      // TODO: Implement CSV parsing
    }

    // প্রতিটি পণ্য সিঙ্ক করুন
    for (const productData of products) {
      await this.prisma.product.upsert({
        where: { tenantId_sku: { tenantId, sku: productData.sku } },
        create: {
          tenantId,
          sku: productData.sku,
          title: productData.title,
          price: productData.price,
          stock: productData.stock,
          description: productData.description,
          images: productData.images || []
        },
        update: {
          title: productData.title,
          price: productData.price,
          stock: productData.stock,
          images: productData.images || []
        }
      });
    }

    this.logger.log(`Synced ${products.length} products for tenant ${tenantId}`);

    return { success: true, syncedCount: products.length };
  }

  // ম্যানুয়াল সিঙ্ক ট্রিগার করুন
  async triggerSync(tenantId: string, sourceUrl: string, sourceType: string) {
    const workflow = await this.prisma.automationWorkflow.create({
      data: {
        tenantId,
        name: `Manual Sync - ${new Date().toISOString()}`,
        type: 'cron',
        trigger: 'manual',
        action: 'sync_product',
        config: { sourceUrl, sourceType },
        isActive: false // একবার চালানোর জন্য
      }
    });

    await this.syncProductsForTenant(tenantId, { sourceUrl, sourceType });

    return { success: true };
  }
}