import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import * as nodemailer from 'nodemailer';

@Injectable()
export class SalesSummaryAutomationService {
  private logger = new Logger('SalesSummaryAutomationService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // প্রতিদিন সকাল 9টায় সেলস রিপোর্ট পাঠান
  @Cron('0 9 * * *')
  async sendDailySalesReport() {
    this.logger.log('Sending daily sales reports...');

    const tenants = await this.prisma.tenant.findMany({
      where: { type: 'admin' }
    });

    for (const tenant of tenants) {
      try {
        await this.generateAndSendReport(tenant.id, tenant.email);
      } catch (err) {
        this.logger.error(`Sales report failed for tenant ${tenant.id}:`, err);
      }
    }
  }

  private async generateAndSendReport(tenantId: string, email: string) {
    // আজকের তারিখ
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const tomorrow = new Date(today);
    tomorrow.setDate(tomorrow.getDate() + 1);

    // আজকের অর্ডার পান
    const orders = await this.prisma.order.findMany({
      where: {
        tenantId,
        createdAt: {
          gte: today,
          lt: tomorrow
        }
      },
      include: { items: true }
    });

    // সেলস সারাংশ গণনা করুন
    const totalOrders = orders.length;
    const totalRevenue = orders.reduce((sum, o) => sum + o.total, 0);
    const totalItems = orders.reduce(
      (sum, o) => sum + o.items.length,
      0
    );

    // রিপোর্ট সংরক্ষণ করুন
    await this.prisma.salesReport.create({
      data: {
        tenantId,
        reportDate: today,
        totalOrders,
        totalRevenue,
        totalItems
      }
    });

    // ইমেইল পাঠান
    const emailHtml = `
<h2>Daily Sales Report</h2>
<p>Date: ${today.toLocaleDateString()}</p>

<table style="border-collapse: collapse; width: 100%;">
  <tr style="background: #f0f0f0;">
    <th style="border: 1px solid #ddd; padding: 10px;">Metric</th>
    <th style="border: 1px solid #ddd; padding: 10px;">Value</th>
  </tr>
  <tr>
    <td style="border: 1px solid #ddd; padding: 10px;">Total Orders</td>
    <td style="border: 1px solid #ddd; padding: 10px;">${totalOrders}</td>
  </tr>
  <tr>
    <td style="border: 1px solid #ddd; padding: 10px;">Total Revenue</td>
    <td style="border: 1px solid #ddd; padding: 10px;">$${totalRevenue.toFixed(2)}</td>
  </tr>
  <tr>
    <td style="border: 1px solid #ddd; padding: 10px;">Total Items Sold</td>
    <td style="border: 1px solid #ddd; padding: 10px;">${totalItems}</td>
  </tr>
</table>
    `;

    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL || 'noreply@store.com',
      to: email,
      subject: `Daily Sales Report - ${today.toLocaleDateString()}`,
      html: emailHtml
    });

    this.logger.log(`Sales report sent for tenant ${tenantId}`);
  }
}